/*
 * Java
 *
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.kf.Kernel;

/**
 * Each Feature has its own pool, which is an extension of the Kernel pool.
 */
public class KFEventGeneratorsPoolFeature extends EventGeneratorsPool {

	private final EventGeneratorsPool kernelPool;

	private KFEventGeneratorsPoolFeature(EventGeneratorsPool kernelPool) {
		super(kernelPool.generators.length);
		this.kernelPool = kernelPool;
	}

	/**
	 * Kernel lazy initialization method for {@link EventGeneratorsPool#SystemPool}
	 */
	public static EventGeneratorsPool newEventGeneratorsPool() {
		return new KFEventGeneratorsPoolFeature(getKernelEventGeneratorsPool());
	}

	private static EventGeneratorsPool getKernelEventGeneratorsPool() {
		Kernel.enter();
		return EventGeneratorsPool.SystemPool;
	}

	@Override
	public <E extends EventGenerator> EventGenerator[] get(Class<E> clazz) {
		EventGenerator[] kernelResult = this.kernelPool.get(clazz);
		EventGenerator[] featureResult = super.get(clazz);
		int kernelResultLength = kernelResult.length;
		int featureResultLength = featureResult.length;
		EventGenerator[] result = new EventGenerator[kernelResultLength + featureResultLength];
		System.arraycopy(kernelResult, 0, result, 0, kernelResultLength);
		System.arraycopy(featureResult, 0, result, kernelResultLength, featureResultLength);
		return result;
	}

	@Override
	public <E extends EventGenerator> E get(Class<E> clazz, int fromIndex) {
		if (fromIndex < this.startId) {
			E result = this.kernelPool.get(clazz, fromIndex);
			if (result != null) {
				return result;
			}
		}
		return super.get(clazz, fromIndex);
	}

	@Override
	public EventGenerator get(int id) {
		if (id < this.startId) {
			return this.kernelPool.get(id);
		} else {
			return super.get(id);
		}
	}
}
