/*
 * Java
 *
 * Copyright 2015-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import com.is2t.tools.ArrayTools;

import ej.microui.MicroUIException;

/**
 * Event generator system pool
 */
public class EventGeneratorsPool {

	public static final int NO_ID = 255; // 2r11111111 (Smalltalk notation for base 2 radix numbers)

	public static EventGeneratorsPool SystemPool = new EventGeneratorsPool(0);

	/* default */ EventGenerator[] generators;
	/* default */ int startId;

	public EventGeneratorsPool(int startId) {
		this.startId = startId;
		this.generators = new EventGenerator[0];
	}

	public int add(EventGenerator gen) {
		int id = gen.getId();
		// check if already in
		if (id != NO_ID) {
			assert sanityCheck(id);
			return id;
		}

		EventGenerator[] generators = this.generators;

		int l = generators.length;
		// scan for an empty place
		int place = NO_ID;// not found
		for (int i = l; --i >= 0;) {
			if (generators[i] == null) {
				// got one...
				place = i;
				break;
			}
		}
		if (place == NO_ID) {// no empty place
			// cannot be bigger than 254 (see NO_ID convention 0xFF)
			if (l >= NO_ID - 1) {
				// too many generators
				throw new MicroUIException(MicroUIException.EVENTGENERATOR_POOL_FULL);
			}
			// build another empty place at the end of the array
			System.arraycopy(generators, 0, generators = new EventGenerator[l + 1], 0, l);
			place = l;
		}
		generators[place] = gen;

		// HERE
		// side effects (these two lines should be atomic... the spec
		// does not mention that it should be so ==> leave it like that)
		gen.setID(place + this.startId);
		this.generators = generators;

		return place;
	}

	public void remove(EventGenerator gen) {
		int id = gen.getId();
		if (id == NO_ID) {
			return;
		}

		assert sanityCheck(id);

		// next two line should be atomic (see HERE comment)
		gen.setID(NO_ID);
		this.generators[id - this.startId] = null;
	}

	private boolean sanityCheck(int id) {
		try {
			Object o = Event.getGenerator(id);
			return o != this;
		} // wrong generator at place id
		catch (IllegalArgumentException ex) {
			return false;
		}
	}

	public EventGenerator[] get() {
		return this.generators;
	}

	public EventGenerator get(int id) {
		EventGenerator ret = this.generators[id - this.startId];
		if (ret == null) {
			throw new IndexOutOfBoundsException(); // according to specification
		}
		return ret;
	}

	public <E extends EventGenerator> E get(Class<E> clazz, int fromIndex) {
		// use local (avoid sync)
		EventGenerator[] pool = this.generators;

		if (pool != null) {
			int poolLength = pool.length;
			for (int i = fromIndex - this.startId - 1; ++i < poolLength;) {
				EventGenerator gen = pool[i];
				if (gen != null && clazz.isAssignableFrom(gen.getClass())) {
					return (E) gen;
				}
			}
		}
		return null;
	}

	public <E extends EventGenerator> EventGenerator[] get(Class<E> clazz) {
		EventGenerator[] pool = this.generators;

		// find all event generators of the given class
		// optimized for a small number of event generators, most likely one (array is
		// resized)
		EventGenerator[] generators = new EventGenerator[0];
		int generatorsPtr = -1;
		for (int i = pool.length; --i >= 0;) {
			EventGenerator gen = pool[i];
			if (gen == null) {
				continue; // skip empty place
			}
			if (clazz.isAssignableFrom(gen.getClass())) {
				generators = (EventGenerator[]) ArrayTools.add(generators, gen, ++generatorsPtr);
			}
		}
		return (EventGenerator[]) ArrayTools.copy(generators, generatorsPtr);
	}
}