/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.MicroUIFactory;
import ej.microui.MicroUIProperties;
import ej.microui.MicroUIPump;

/**
 * microUI-API
 */
public abstract class EventGenerator {

	private @Nullable EventHandler eventHandler;

	/**
	 * [KF] The default event handler is the first event handler attached to this event generator.
	 */
	private @Nullable EventHandler defaultEventHandler;

	/**
	 * from -1 to 254 (255 is reserved)
	 *
	 * @see EventGeneratorsPool#NO_ID
	 */
	private int id;

	/**
	 * microUI-API
	 */
	public EventGenerator() {
		this.id = EventGeneratorsPool.NO_ID; // means "not a system generator"
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int addToSystemPool() {
		return EventGeneratorsPool.SystemPool.add(this);
	}

	/**
	 * microUI-API
	 */
	public void removeFromSystemPool() {
		EventGeneratorsPool.SystemPool.remove(this);
	}

	/**
	 * microUI-API
	 *
	 * @param eventHandler
	 *            microUI-API
	 * @throws SecurityException
	 *             microUI-API
	 */
	public void setEventHandler(@Nullable EventHandler eventHandler) throws SecurityException {
		MicroUIFactory.getInstance().setEventHandler(this, eventHandler);
	}

	/**
	 * microUI-API
	 *
	 * @param id
	 *            microUI-API
	 * @return microUI-API
	 */
	public static EventGenerator get(int id) {
		MicroUI.checkRunning();
		return EventGeneratorsPool.SystemPool.get(id);
	}

	/**
	 * microUI-API
	 *
	 * @param clazz
	 *            microUI-API
	 * @param fromIndex
	 *            microUI-API
	 * @return microUI-API
	 */
	public static @Nullable <E extends EventGenerator> E get(Class<E> clazz, int fromIndex) {
		assert clazz != null;
		MicroUI.checkRunning();
		return EventGeneratorsPool.SystemPool.get(clazz, fromIndex);
	}

	/**
	 * microUI-API
	 *
	 * @param clazz
	 *            microUI-API
	 * @return microUI-API
	 */
	public static <E extends EventGenerator> EventGenerator[] get(Class<E> clazz) {
		assert clazz != null;
		MicroUI.checkRunning();
		return EventGeneratorsPool.SystemPool.get(clazz);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public final int getId() {
		return this.id;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public @Nullable EventHandler getEventHandler() {
		return this.eventHandler;
	}

	/**
	 * microUI-API Gets the event type associated with the event generator
	 *
	 * @return the event type
	 */
	public abstract int getEventType();

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 */
	protected void sendEvent(int event) {
		sendEvent(event, getEventHandler());
	}

	/**
	 * Used by subclass to give a MicroUI event to the given event handler.
	 * <p>
	 * The handler might be null (not standard use case).
	 *
	 * @param event
	 *            the MicroUI event to send
	 * @param eventHandler
	 *            the receiver
	 */
	protected void sendEvent(int event, @Nullable EventHandler eventHandler) {
		MicroUIPump.callHandler(event, eventHandler);
	}

	/**
	 * Asks the generator to convert the <code>inputEvent</code> in order to send a MicroUI event to the given event
	 * handler. This call may do side-effects on the receiver (call {@link MicroUIPump#read()}).
	 *
	 * @param pump
	 *            the pump where read additional events
	 * @param inputEvent
	 *            the input event to convert into a MicroUI event
	 * @param eventHandler
	 *            the MicroUI event receiver
	 * @throws InterruptedException
	 *             see {@link MicroUIPump#read()}
	 */
	public void convert(MicroUIPump pump, int inputEvent, EventHandler eventHandler) throws InterruptedException {
		// let's the sub classes do their work
		// not abstract to prevent user's event generators compiletime errors
	}

	/**
	 * [KF] Checks permissions
	 */
	public void checkEventHandlerPermission() {
		if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = System.getSecurityManager();
			if (sm != null) {
				sm.checkPermission(new EventPermission(this, EventPermission.PERMISSION_SET));
			}
		}
	}

	/**
	 * [KF] Replaces the event handler.
	 *
	 * @param handler
	 *            the new handler.
	 */
	public void setEventHandler0(@Nullable EventHandler handler) {
		this.eventHandler = handler;
	}

	/**
	 * [KF] Sets the default event handler if not already set.
	 *
	 * @param handler
	 *            the new handler.
	 */
	public void setDefaultEventHandler0(@Nullable EventHandler handler) {
		if (this.defaultEventHandler == null) {
			this.defaultEventHandler = handler;
		}
	}

	/**
	 * [KF] Gets the default handler.
	 *
	 * @return the default handler.
	 */
	public @Nullable EventHandler getDefaultEventHandler() {
		return this.defaultEventHandler;
	}

	/**
	 * Sets the identifier.
	 *
	 * @param id
	 *            the new id
	 */
	public void setID(int id) {
		this.id = id;
	}
}
