/*
 * Java
 *
 * Copyright 2010-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.microui.MicroUI;
import ej.microui.SystemMicroUI;

/**
 * microUI-API
 */

public class Event {

	private static final int TYPE_MASK = 0xFF;
	private static final int GENERATOR_MASK = 0xFF;
	private static final int DATA_MASK = 0xFFFF;

	private static final int GENERATOR_SHIFT = 16;
	private static final int DATA_SHIFT = 0;

	/**
	 * microUI-API
	 */
	private Event() {
	}

	/**
	 * microUI-API
	 *
	 * @param type
	 *            microUI-API
	 * @param gen
	 *            microUI-API
	 * @param data
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int buildEvent(int type, EventGenerator gen, int data) {
		assert gen != null;

		// retrieve the generator ID; implementation ensures this ID is smaller than
		// GENERATOR_MASK
		int genID = gen.getId();
		assert checkGeneratorID(genID);

		return SystemMicroUI.buildEvent(type & TYPE_MASK, (genID << GENERATOR_SHIFT) | (data & DATA_MASK));
	}

	private static boolean checkGeneratorID(int genID) {
		return genID >= 0 && genID <= GENERATOR_MASK;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getType(int event) {
		// global MicroUI event specification (not reserved to EventGenerator events)
		return SystemMicroUI.getType(event);
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getData(int event) {
		return (event >>> DATA_SHIFT) & DATA_MASK;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getGeneratorId(int event) {
		return (event >>> GENERATOR_SHIFT) & GENERATOR_MASK;
	}

	/**
	 * microUI-API
	 *
	 * @param event
	 *            microUI-API
	 * @return microUI-API
	 */
	public static EventGenerator getGenerator(int event) {
		try {
			return EventGenerator.get(getGeneratorId(event));
		} catch (IndexOutOfBoundsException e) {
			// check running in exception to prevent to check it at each call to
			// getGenerator()
			MicroUI.checkRunning();
			throw new IllegalArgumentException();
		}
	}
}
