/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.microui.MicroUI;

public abstract class WaitForEventRunnable implements Runnable {

	/**
	 * This state indicates the event has not yet been processed
	 */
	private static final int PENDING = 0;

	/**
	 * This state indicates the event has been successfully processed
	 */
	private static final int EXECUTED = 1;

	/**
	 * This state indicates the event has been canceled
	 */
	private static final int CANCELED = 2;

	private int state = PENDING;

	@Override
	public final synchronized void run() {
		if (this.state == PENDING) {
			executeEvent();
			this.state = EXECUTED;
			notifyWaitingThread();
		} else {
			// the event has been cancelled. Silently do nothing.
		}
	}

	/**
	 * Notifies the waiting thread. By default notifies only one thread.
	 */
	protected synchronized void notifyWaitingThread() {
		notify();
	}

	/**
	 * Wait until this runnable has been processed by the DisplayPump.
	 *
	 * @return <code>true</code> if the execution has been successfully terminated, <code>false</code> if it has been
	 *         canceled.
	 */
	public synchronized boolean waitForExecution() {
		while ((this.state == PENDING) && MicroUI.isStarted()) {
			try {
				wait();
			} catch (InterruptedException ex) {
				continue;
			}
		}
		return this.state == EXECUTED;
	}

	/**
	 * Cancel the event execution. The Thread blocked will be resumed and {@link #waitForExecution()} will return
	 * <code>false</code>.
	 */
	public synchronized void cancel() {
		if (this.state == PENDING) {
			cancelEvent();
			this.state = CANCELED;
			notifyWaitingThread();
		}
	}

	/**
	 * Execute the event.
	 */
	protected abstract void executeEvent();

	/**
	 * Cancels the event.
	 */
	protected abstract void cancelEvent();
}
