/*
 * Java
 *
 * Copyright 2019-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.bon.ByteArray;

/**
 * This class holds a byte array used by graphical engine to store the image metadata.
 * <p>
 * Actually the graphical engine is using two structs: one for immutable image and one for mutable image (which extends
 * first struct and adding some fields). This class should "represents" these both structs (not performed yet).
 * <p>
 * On simulator there are some restrictions: the fields are stored in big endian by graphical engine (== S3 endianness)
 * but the API {@link ByteArray#readInt(byte[], int)} reads an integer in platform endianness (often little endian). So
 * a patch is performed in {@link Tools#getInt(byte[], int)} (and others methods) for simulator.
 *
 */
public class SImageMetadata {

	/**
	 * Opacity mask
	 */
	public static final int ALPHA_MASK = 0xff000000;

	/**
	 * Upper part: immutable image
	 */
	private static final int OFFSET_I_OFFSCREEEN_ID = 0;
	private static final int OFFSET_C_WIDTH = OFFSET_I_OFFSCREEEN_ID + 4;
	private static final int OFFSET_C_HEIGHT = OFFSET_C_WIDTH + 2;
	private static final int OFFSET_C_MEMWH = OFFSET_C_HEIGHT + 2;
	private static final int OFFSET_B_FORMAT = OFFSET_C_MEMWH + 2;
	private static final int OFFSET_B_FLAGS = OFFSET_B_FORMAT + 1;
	private static final int SD_IMMUTABLE_SIZE = OFFSET_B_FLAGS + 1;

	/**
	 * Lower part: mutable image
	 */
	private static final int OFFSET_I_FOREGROUND_COLOR = SD_IMMUTABLE_SIZE + 0;
	private static final int OFFSET_I_BACKGROUND_COLOR = OFFSET_I_FOREGROUND_COLOR + 4;
	private static final int OFFSET_S_CLIPX1 = OFFSET_I_BACKGROUND_COLOR + 4;
	private static final int OFFSET_S_CLIPY1 = OFFSET_S_CLIPX1 + 2;
	private static final int OFFSET_S_CLIPX2 = OFFSET_S_CLIPY1 + 2;
	private static final int OFFSET_S_CLIPY2 = OFFSET_S_CLIPX2 + 2;
	private static final int OFFSET_I_DRAWING_LOG_FLAGS = OFFSET_S_CLIPY2 + 2;
	private static final int OFFSET_B_DRAWER = OFFSET_I_DRAWING_LOG_FLAGS + 4;
	private static final int OFFSET_C_ELLIPSIS = OFFSET_B_DRAWER + 2;
	private static final int SD_MUTABLE_SIZE = OFFSET_C_ELLIPSIS + 2;

	/**
	 * sync with SImageMetadata.icetea and sub struct
	 * <p>
	 */
	private static final int FLAG_MUTABLE = 1 << 0; // true when this targets a mutable image or display
	private static final int FLAG_FREED = 1 << 1; // true when the image data memory has to be freed (runtime allocator)
	private static final int FLAG_MAIN_SCREEN = 1 << 2; // true when this targets a main screen (not an image)
	private static final int FLAG_NO_HEADER = 1 << 3; // true when there is no header before the pixels array
	private static final int FLAG_CLOSED = 1 << 4; // true when the image has been closed
	private static final int FLAG_BACKCOLOR = 1 << 6; // true if there is a background color

	private SImageMetadata() {
		// private constructor
	}

	/**
	 * Allocates a byte array to store the metadata of a mutable image or of an immutable image.
	 *
	 * @param mutable
	 *            true to allocate a byte array where storing the mutable image metadata
	 * @return the allocated byte array
	 */
	public static byte[] allocate(boolean mutable) {
		return new byte[mutable ? SD_MUTABLE_SIZE : SD_IMMUTABLE_SIZE];
	}

	/**
	 * Internal image reference in the graphical engine
	 *
	 * @return an integer which represents the image in the graphical engine.
	 */
	public static int getImageReference(byte[] sd) {
		return Tools.getInt(sd, OFFSET_I_OFFSCREEEN_ID);
	}

	/**
	 * Tells if an image has to be freed in native world
	 *
	 * @return true if the image has to be freed (allocated in images heap).
	 *
	 * @deprecated Kept for backward compatibility (widget2 over UI3). Task M0092MEJAUI-2065.
	 */
	@Deprecated
	public static boolean isFreed(byte[] sd) {
		return isFlagSet(sd, FLAG_FREED);
	}

	public static boolean hasBackgroundColor(byte[] sd) {
		return isFlagSet(sd, FLAG_BACKCOLOR);
	}

	public static int getEllipsis(byte[] sd) {
		return Tools.getChar(sd, OFFSET_C_ELLIPSIS);
	}

	public static int getWidth(byte[] sd) {
		return Tools.getChar(sd, OFFSET_C_WIDTH);
	}

	public static int getHeight(byte[] sd) {
		return Tools.getChar(sd, OFFSET_C_HEIGHT);
	}

	public static int getForegroundColor(byte[] sd) {
		return Tools.getInt(sd, OFFSET_I_FOREGROUND_COLOR);
	}

	public static int getBackgroundColor(byte[] sd) {
		return Tools.getInt(sd, OFFSET_I_BACKGROUND_COLOR);
	}

	/**
	 * This method MUST ONLY BE called by {@link GraphicsContext#updateJavaClip()}
	 */
	public static int getClipX1(byte[] sd) {
		return Tools.getShort(sd, OFFSET_S_CLIPX1);
	}

	/**
	 * This method MUST ONLY BE called by {@link GraphicsContext#updateJavaClip()}
	 */
	public static int getClipY1(byte[] sd) {
		return Tools.getShort(sd, OFFSET_S_CLIPY1);
	}

	private static int getClipX2(byte[] sd) {
		return Tools.getShort(sd, OFFSET_S_CLIPX2);
	}

	private static int getClipY2(byte[] sd) {
		return Tools.getShort(sd, OFFSET_S_CLIPY2);
	}

	/**
	 * This method MUST ONLY BE called by {@link GraphicsContext#updateJavaClip()}
	 */
	public static int getClipWidth(byte[] sd) {
		return getClipX2(sd) - getClipX1(sd) + 1;
	}

	/**
	 * This method MUST ONLY BE called by {@link GraphicsContext#updateJavaClip()}
	 */
	public static int getClipHeight(byte[] sd) {
		return getClipY2(sd) - getClipY1(sd) + 1;
	}

	private static boolean isFlagSet(byte[] sd, int flag) {
		return (sd[OFFSET_B_FLAGS] & flag) == flag;
	}
}
