/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayInputStreamUtils;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;

import com.is2t.microbsp.microui.natives.NSystemDisplay;
import com.is2t.microbsp.microui.natives.NTools;
import com.is2t.vm.support.io.MemoryInputStream;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.Log;
import ej.microui.MicroUIException;
import ej.microui.MicroUIProperties;

/**
 * microUI-API
 */
public class ResourceImage extends Image implements Closeable {

	/**
	 * microUI-API
	 */
	/*
	 * Same constants in Format.iceTea
	 */
	public enum OutputFormat {
		/**
		 * microUI-API
		 */
		ARGB8888(2),
		/**
		 * microUI-API
		 */
		RGB888(3),
		/**
		 * microUI-API
		 */
		RGB565(4),
		/**
		 * microUI-API
		 */
		ARGB1555(5),
		/**
		 * microUI-API
		 */
		ARGB4444(6),
		/**
		 * microUI-API
		 */
		A8(8);

		private final int val;

		OutputFormat(int val) {
			this.val = val;
		}

		/**
		 * microUI-API
		 *
		 * Returns the SNI context data of this format.
		 *
		 * @return the value.
		 */
		public byte getSNIContext() {
			return (byte) this.val;
		}
	}

	/**
	 *
	 */
	ResourceImage(byte[] sd, int logOpen, int logType) {
		super(sd, logOpen, logType);
	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean canLoadImage(String name) {
		name = checkImagePathPrerequisites(name, ImagePermission.PERMISSION_CAN_LOAD);
		return canLoadImage(name, false, false) || canLoadImage(name, true, false) || canLoadImage(name, false, true)
				|| canLoadImage(name, true, true);
	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @return microUI-API
	 */
	public static ResourceImage loadImage(String name) {
		return loadImage(name, NSystemDisplay.DEFAULT_FORMAT, ImagePermission.PERMISSION_LOAD);
	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @param format
	 *            microUI-API
	 * @return microUI-API
	 */
	public static ResourceImage loadImage(String name, @Nullable OutputFormat format) {
		return loadImage(name, getFormat(format), ImagePermission.PERMISSION_LOAD_FORMAT);
	}

	private static ResourceImage loadImage(String name, int format, String permission) {

		name = checkImagePathPrerequisites(name, permission);

		logCreateImageStart(Log.TYPE_OPEN_IMAGE_FROM_PATH);

		byte[] sd = SImageMetadata.allocate(false);

		// 1- try to load an internal raw image
		ResourceImage image = loadImage(name, sd, format, false, false, Log.IMAGE_CARACTERISTICS_INTERNAL_RAW);
		if (image != null) {
			// done!
			return image;
		}

		// 2- try to load an internal dynamic image
		image = loadImage(name, sd, format, true, false, Log.IMAGE_CARACTERISTICS_INTERNAL_DYNAMIC);
		if (image != null) {
			// done!
			return image;
		}

		// 3- try to load an external raw image
		image = loadImage(name, sd, format, false, true, Log.IMAGE_CARACTERISTICS_EXTERNAL_RAW);
		if (image != null) {
			// done!
			return image;
		}

		// 4- try to load an external dynamic image
		image = loadImage(name, sd, format, true, true, Log.IMAGE_CARACTERISTICS_EXTERNAL_DYNAMIC);
		if (image != null) {
			// done!
			return image;
		}

		throw new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH, name);
	}

	private static String getImagePath(String name, boolean dynamic, boolean externalResource) {
		String fullname = dynamic ? name : NTools.getRawImagePath(name);
		return externalResource ? NTools.getExternalImagePath(fullname) : fullname;
	}

	private static boolean canLoadImage(String name, boolean dynamic, boolean externalResource) {
		String fullname = getImagePath(name, dynamic, externalResource);
		return NSystemDisplay.NO_ERROR == NSystemDisplay.loadImageFromPath(fullname.getBytes(), fullname.length(),
				dynamic, externalResource, false, NSystemDisplay.DEFAULT_FORMAT, null);
	}

	private static @Nullable ResourceImage loadImage(String name, byte[] sd, int format, boolean dynamic,
			boolean externalResource, int logType) {

		String fullname = getImagePath(name, dynamic, externalResource);
		int errorCode = NSystemDisplay.loadImageFromPath(fullname.getBytes(), fullname.length(), dynamic,
				externalResource, false, format, sd);

		switch (errorCode) {
		case NSystemDisplay.NO_ERROR:
			return new ResourceImage(sd, Log.TYPE_OPEN_IMAGE_FROM_PATH, logType);
		case NSystemDisplay.INVALID_PATH:
			// let caller manage the error
			return null;
		default:
			// file found but native exception: throw exception
			NTools.checkError(errorCode, name);
			return null; // never here
		}
	}

	/**
	 * microUI-API
	 *
	 * @param stream
	 *            microUI-API
	 * @param size
	 *            microUI-API
	 * @return microUI-API
	 */
	public static ResourceImage loadImage(InputStream stream, int size) {
		return loadImage(stream, size, NSystemDisplay.DEFAULT_FORMAT, ImagePermission.PERMISSION_LOAD_INPUTSTREAM);
	}

	/**
	 * microUI-API
	 *
	 * @param stream
	 *            microUI-API
	 * @param size
	 *            microUI-API
	 * @param format
	 *            microUI-API
	 * @return microUI-API
	 */
	public static ResourceImage loadImage(InputStream stream, int size, @Nullable OutputFormat format) {
		return loadImage(stream, size, getFormat(format), ImagePermission.PERMISSION_LOAD_INPUTSTREAM_FORMAT);
	}

	private static ResourceImage loadImage(InputStream stream, final int size, int format, String permission) {

		assert stream != null;
		checkPrerequisites(permission);

		logCreateImageStart(Log.TYPE_OPEN_IMAGE_FROM_INPUTSTREAM);

		byte[] sd = SImageMetadata.allocate(false);
		int logType = Log.IMAGE_CARACTERISTICS_GENERICINPUTSTREAM;
		int errorCode = NSystemDisplay.NO_ERROR; // prevent java compiler issue

		try {

			if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {

				if (stream instanceof MemoryInputStream) {
					// On emb, a memory input stream targets a fixed memory region (immutable). On
					// sim, this notion does not exist. MemoryInputStream is available in
					// edc-internal. On sim, no object is an instance of MemoryInputStream. But if
					// in the future this use case appear, HIL protocol must be updated to authorize
					// a MemoryInputStream as a field

					logType = Log.IMAGE_CARACTERISTICS_MEMORYINPUTSTREAM;
					errorCode = NSystemDisplay.loadImageFromMIS((MemoryInputStream) stream, size, format, sd);

				} else if (stream instanceof ByteArrayInputStream) {
					// we can directly use the byte array

					byte[] data = ByteArrayInputStreamUtils.getByteArray((ByteArrayInputStream) stream);
					int offset = ByteArrayInputStreamUtils.getOffset((ByteArrayInputStream) stream);
					stream.skip(size); // skip bytes read in native side, can throw an IOE

					logType = Log.IMAGE_CARACTERISTICS_BYTEARRAYINPUTSTREAM;
					errorCode = NSystemDisplay.loadImageFromBytes(data, offset, size, format, sd);
				}
			}

			if (Log.IMAGE_CARACTERISTICS_GENERICINPUTSTREAM == logType) {
				// we have to read the stream

				DataInputStream dis = stream instanceof DataInputStream ? (DataInputStream) stream
						: new DataInputStream(stream);
				byte[] data = new byte[size];
				try {
					dis.readFully(data); // can throw an IOE
				} finally {
					if (!(stream instanceof DataInputStream)) {
						// have to close the temporary dis
						dis.close();
					}
				}

				errorCode = NSystemDisplay.loadImageFromBytes(data, 0, size, format, sd);
			}
		} catch (IOException e) {
			throw new MicroUIException(MicroUIException.RESOURCE_INVALID_FILE);
		}

		NTools.checkError(errorCode, null);
		return new ResourceImage(sd, Log.TYPE_OPEN_IMAGE_FROM_INPUTSTREAM, logType);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean isClosed() {
		return this.sd == null;
	}

	@Override
	public void close() {
		byte[] data = this.sd;
		if (data != null) {
			this.sd = null;
			NSystemDisplay.closeOffScreen(data);
		}
	}

	/**
	 * Return integer format from {@link OutputFormat}
	 */
	private static int getFormat(@Nullable OutputFormat format) {
		return format == null ? NSystemDisplay.DEFAULT_FORMAT : format.val;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	@Override
	public byte[] getSNIContext() {
		if (this.sd == null) {
			// image has been closed
			throw new MicroUIException(MicroUIException.RESOURCE_CLOSED);
		}
		return this.sd;
	}
}
