/*
 * Java
 *
 * Copyright 2017-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.MicroUIProperties;

/**
 * This class associates a string with a font. The string rendering is accelerated because the couple string/font is
 * fixed.
 */
public class RenderableString {

	/* package */ final String string;
	/* package */ final byte[] sniContext;
	/* package */ final Font font;
	/* package */ final int width;
	@Nullable
	private char[] charArray; // null init; useless until first call to getCharArray()

	/**
	 * microUI-API
	 *
	 * @param string
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 */
	public RenderableString(String string, Font font) {
		assert string != null && font != null;

		this.string = string;
		this.font = font;

		char[] chars;
		int offset;
		int length;
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			chars = this.string.chars;
			offset = this.string.offset;
			length = this.string.length;
		} else {
			chars = Font.getStringChars(string);
			offset = 0;
			length = chars.length;
		}
		this.sniContext = font.allocateRenderableStringSNIContext(chars, offset, length);
		this.width = PainterNatives.initializeRenderableStringSNIContext(chars, offset, length, font.getSNIContext(),
				this.sniContext);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public String getString() {
		return this.string;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public char[] getCharArray() {
		// useless to create this array systematically

		char[] array = this.charArray;

		// first check prevents to sync at each call
		if (array == null) {

			synchronized (this) {

				// second check is required when two threads are trying to get array at same
				// time (for the first time only)
				array = this.charArray;
				if (array == null) {
					this.charArray = this.string.toCharArray();
					array = this.charArray;
				}
			}
		}

		return array;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 * @deprecated useless on the application side.
	 */
	@Deprecated
	public int[] getIndices() {
		return new int[0];
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public Font getFont() {
		return this.font;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getWidth() {
		return this.width;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getHeight() {
		return this.font.getHeight();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public byte[] getSNIContext() {
		return this.sniContext;
	}
}
