/*
 * Java
 *
 * Copyright 2008-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;
import com.is2t.tools.ArrayTools;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microui.MicroUIProperties;

class RasterFont extends Font {

	/**
	 * This default index is used when the string indexes have not been computed yet.
	 */
	private static final byte INVALID_INDEX = -1;

	public static final int STYLE_PLAIN = 0;

	public static final int STYLE_BOLD = 1;

	public static final int STYLE_ITALIC = 1 << 1;

	private static final int SUFFIX_LENGTH = "_raw".length(); //$NON-NLS-1$

	private static @Nullable RasterFont[] fonts; // use getFonts()

	/**
	 * List of all fonts loaded at same time than current font. This object is unique for all fonts.
	 */
	private final String list;

	/**
	 * Start index of path of this font in the previous list
	 */
	private final char pathStartIndex;

	/**
	 * End index of path of this font in the previous list
	 */
	private final char pathEndIndex;

	private final char height;
	private final char baseline;
	private final int[] identifiers;
	private final byte style;

	private RasterFont(byte[] nativeData, String list, int pathStartIndex, int pathEndIndex) {
		super(nativeData);
		this.list = list;
		this.pathStartIndex = (char) pathStartIndex;
		this.pathEndIndex = (char) pathEndIndex;

		byte[] sniContext = getSNIContext();
		this.height = FontData.getHeight(sniContext);
		this.baseline = FontData.getBaseline(sniContext);
		this.identifiers = FontData.getIdentifiers(sniContext);

		int styleFlags = STYLE_PLAIN;
		if (FontData.isBold(sniContext)) {
			styleFlags |= STYLE_BOLD;
		}
		if (FontData.isItalic(sniContext)) {
			styleFlags |= STYLE_ITALIC;
		}
		this.style = (byte) styleFlags;
	}

	@Override
	public int[] getIdentifiers() {
		// spec does not specify if we have to return a copy
		return this.identifiers;
	}

	@Override
	public boolean isIdentifierSupported(int identifier) {
		int[] identifiers = getIdentifiers();
		for (int i = identifiers.length; --i >= 0;) {
			if (identifiers[i] == identifier) {
				return true;
			}
		}
		return false;
	}

	@Override
	public int getStyle() {
		return this.style & (STYLE_BOLD | STYLE_ITALIC | STYLE_PLAIN);
	}

	@Override
	public String getDescriptor() {
		int filenameStartIndex = this.list.lastIndexOf('/', this.pathEndIndex) + 1;
		int filenameEndIndex = this.list.lastIndexOf('.', this.pathEndIndex);
		return this.list.substring(filenameStartIndex, filenameEndIndex);
	}

	private boolean isSamePath(String path) {
		int pathLength = path.length();
		return pathLength == (this.pathEndIndex - this.pathStartIndex - SUFFIX_LENGTH) // end index is the separator
				&& this.list.regionMatches(this.pathStartIndex, path, 0, pathLength);
	}

	@Override
	public boolean isPlain() {
		// the style is Plain if no BOLD and no Italic (see getStyle())
		return (this.style & (STYLE_BOLD | STYLE_ITALIC)) == 0;
	}

	@Override
	public boolean isBold() {
		return (this.style & STYLE_BOLD) != 0;
	}

	@Override
	public boolean isItalic() {
		return (this.style & STYLE_ITALIC) != 0;
	}

	@Override
	public boolean isMonospaced() {
		return FontData.isMonospaced(getSNIContext());
	}

	public static RasterFont getDefaultFont() {
		try {
			return getFontsInternal()[0];
		} catch (Exception e) {
			throw new MicroUIException(MicroUIException.NO_FONT);
		}
	}

	public static RasterFont[] getAllFonts() {
		// copy the array of fonts
		RasterFont[] fonts = getFonts();
		int nbFonts = fonts.length;
		RasterFont[] copy = new RasterFont[nbFonts];
		System.arraycopy(fonts, 0, copy, 0, nbFonts);
		return copy;
	}

	public static RasterFont getFont(String path) {
		assert path != null;
		RasterFont[] fonts = getFonts();
		String p = MicroUIException.checkResourcePath(path);
		for (RasterFont font : fonts) {
			if (font.isSamePath(p)) {
				return font;
			}
		}
		throw new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH, p);
	}

	@Override
	public int getHeight() {
		return this.height;
	}

	@Override
	public int getBaselinePosition() {
		return this.baseline;
	}

	private static synchronized RasterFont[] getFonts() {
		// indirection known by KF, see KFDisplayFont
		return getFontsInternal();
	}

	/**
	 * Lazy initialization of Fonts.
	 * <p>
	 * Synchronization must be ensured by the caller.
	 */
	/* default */ static RasterFont[] getFontsInternal() {
		if (fonts == null) {
			// No injection - default case
			RasterFont.fonts = initializeAllFonts();
		}
		return fonts;
	}

	/* default */ static RasterFont[] initializeAllFonts() {
		return addFonts(Constants.getString(MicroUIProperties.FONTS_LIST), new RasterFont[0]);
	}

	/* default */ static RasterFont[] addFonts(final String list, RasterFont[] fonts) {
		// the try-catch block should be written inside the if-constants block. But
		// the issue M0081MEJA-842 prevents it.
		// 96 bytes in flash can be won moving this try-catch.
		try {
			checkPrerequisites(FontPermission.PERMISSION_ADD);
		} catch (SecurityException e) {
			// Not allowed to add font - silently complete
			return fonts;
		}

		// add all user fonts
		int index = 0;
		int listLength = list.length();
		while (index < listLength) {
			// get the next index of ':'
			int sepIndex = list.indexOf(MicroUIProperties.FONTS_LIST_SEPARATOR, index);

			// it was the last identifier
			if (sepIndex == -1) {
				sepIndex = listLength;
			}

			// add all MicroUI fonts for this path
			fonts = addFont(list, fonts, index, sepIndex);

			// update index
			index = sepIndex + 1;
		}

		return fonts;
	}

	private static RasterFont[] addFont(final String list, RasterFont[] microUIFonts, int pathStartIndex,
			int pathEndIndex) {
		// trim path manually
		while (pathStartIndex < pathEndIndex && list.charAt(pathStartIndex) <= ' ') {
			pathStartIndex++;
		}
		while (pathStartIndex < pathEndIndex && list.charAt(pathEndIndex - 1) <= ' ') {
			pathEndIndex--;
		}

		if (pathStartIndex >= pathEndIndex) {
			return microUIFonts; // empty
		}

		String path = list.substring(pathStartIndex, pathEndIndex);

		if (path.charAt(0) != '/') {
			MicroUI.MicroUI.errorLog(new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH, path));
			return microUIFonts; // empty
		}

		try {
			// add NULL character at the end of string to be able load an external font
			String nativePath = path + '\0';

			// load font and retrieve some font characteristics
			byte[] nativeFontData = FontData.allocate();
			checkFontLoadingError(
					NSystemDisplay.addInternalFont(nativePath.getBytes(), nativePath.length(), nativeFontData));

			// create a font object for the ejf file
			RasterFont font = new RasterFont(nativeFontData, list, pathStartIndex, pathEndIndex);

			microUIFonts = (RasterFont[]) ArrayTools.add(microUIFonts, font);
		} catch (Exception e) {
			// just print a warning and try to load remaining fonts
			MicroUI.MicroUI.errorLog(e);
		}

		return microUIFonts;
	}

	/**
	 * Check the error returned during a font loading: addFont, getIdentifiers etc.
	 *
	 * @param ret
	 *
	 * @return
	 */
	/* default */ static int checkFontLoadingError(int ret) {
		if (ret == NSystemDisplay.INVALID_PATH) {
			throw new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH);
		} else if (ret == NSystemDisplay.INVALID_FILE) {
			throw new MicroUIException(MicroUIException.RESOURCE_INVALID_FILE);
		}
		return ret;
	}

	@Override
	protected byte[] allocateRenderableStringSNIContext(char[] chars, int offset, int length) {
		// array of indexes (an integer per character)
		byte[] sniContext = new byte[length * Integer.SIZE / Byte.SIZE];
		int sniContextLength = sniContext.length;
		for (int i = 0; i < sniContextLength; i++) {
			sniContext[i] = INVALID_INDEX;
		}
		return sniContext;
	}
}
