/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.tools.ArrayTools;

import ej.bon.Constants;
import ej.microui.MicroUIProperties;

/**
 * microUI-API
 */
public class Painter {

	private Painter() {
		// static methods only
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void writePixel(GraphicsContext gc, int x, int y) {
		assert gc != null;
		PainterNatives.writePixel(gc.getSNIContext(), x + gc.translateX, y + gc.translateY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 */
	public static void drawHorizontalLine(GraphicsContext gc, int x, int y, int length) {
		assert gc != null;
		// if width == 0, nothing to draw (test performed on native side)
		PainterNatives.drawHorizontalLine(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, length);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 */
	public static void drawVerticalLine(GraphicsContext gc, int x, int y, int length) {
		assert gc != null;
		// if height == 0, nothing to draw (test performed on native side)
		PainterNatives.drawVerticalLine(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, length);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param startX
	 *            microUI-API
	 * @param startY
	 *            microUI-API
	 * @param endX
	 *            microUI-API
	 * @param endY
	 *            microUI-API
	 */
	public static void drawLine(GraphicsContext gc, int startX, int startY, int endX, int endY) {
		assert gc != null;
		PainterNatives.drawLine(gc.getSNIContext(), startX + gc.translateX, startY + gc.translateY,
				endX + gc.translateX, endY + gc.translateY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public static void drawRectangle(GraphicsContext gc, int x, int y, int width, int height) {
		assert gc != null;
		// if width == 0 || height == 0, nothing to draw (test performed on native side)
		PainterNatives.drawRectangle(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public static void fillRectangle(GraphicsContext gc, int x, int y, int width, int height) {
		assert gc != null;
		PainterNatives.fillRectangle(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param cornerEllipseWidth
	 *            microUI-API
	 * @param cornerEllipseHeight
	 *            microUI-API
	 */
	public static void drawRoundedRectangle(GraphicsContext gc, int x, int y, int width, int height,
			int cornerEllipseWidth, int cornerEllipseHeight) {
		assert gc != null;
		PainterNatives.drawRoundedRectangle(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, width, height,
				cornerEllipseWidth, cornerEllipseHeight);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param cornerEllipseWidth
	 *            microUI-API
	 * @param cornerEllipseHeight
	 *            microUI-API
	 */
	public static void fillRoundedRectangle(GraphicsContext gc, int x, int y, int width, int height,
			int cornerEllipseWidth, int cornerEllipseHeight) {
		assert gc != null;
		PainterNatives.fillRoundedRectangle(gc.getSNIContext(), x + gc.translateX, y + gc.translateY, width, height,
				cornerEllipseWidth, cornerEllipseHeight);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 */
	public static void drawCircleArc(GraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle) {
		assert gc != null;
		PainterNatives.drawCircleArc(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, diameter, startAngle,
				arcAngle);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 */
	public static void drawEllipseArc(GraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		assert gc != null;
		PainterNatives.drawEllipseArc(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, width, height,
				startAngle, arcAngle);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 */
	public static void fillCircleArc(GraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle) {
		assert gc != null;
		PainterNatives.fillCircleArc(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, diameter, startAngle,
				arcAngle);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 */
	public static void fillEllipseArc(GraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		assert gc != null;
		PainterNatives.fillEllipseArc(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, width, height,
				startAngle, arcAngle);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 */
	public static void drawCircle(GraphicsContext gc, int x, int y, int diameter) {
		assert gc != null;
		PainterNatives.drawCircle(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, diameter);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 */
	public static void fillCircle(GraphicsContext gc, int x, int y, int diameter) {
		assert gc != null;
		PainterNatives.fillCircle(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, diameter);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public static void drawEllipse(GraphicsContext gc, int x, int y, int width, int height) {
		assert gc != null;
		PainterNatives.drawEllipse(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public static void fillEllipse(GraphicsContext gc, int x, int y, int width, int height) {
		assert gc != null;
		PainterNatives.fillEllipse(gc.getSNIContext(), gc.translateX + x, gc.translateY + y, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xSrc
	 *            microUI-API
	 * @param ySrc
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param xDest
	 *            microUI-API
	 * @param yDest
	 *            microUI-API
	 */
	public static void drawDisplayRegion(GraphicsContext gc, int xSrc, int ySrc, int width, int height, int xDest,
			int yDest) {
		assert gc != null;
		drawRegion(gc, Display.getDisplay().getGraphicsContext().getSNIContext(), xSrc, ySrc, width, height, xDest,
				yDest, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xSrc
	 *            microUI-API
	 * @param ySrc
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param xDest
	 *            microUI-API
	 * @param yDest
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawDisplayRegion(GraphicsContext gc, int xSrc, int ySrc, int width, int height, int xDest,
			int yDest, int alpha) {
		assert gc != null;
		drawRegion(gc, Display.getDisplay().getGraphicsContext().getSNIContext(), xSrc, ySrc, width, height, xDest,
				yDest, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param str
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void drawString(GraphicsContext gc, String str, Font font, int x, int y) {
		assert gc != null && str != null && font != null;
		int l = str.length(); // can throw NPE (spec)
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			font.drawChars(gc, str.chars, str.offset, l, x + gc.translateX, y + gc.translateY);
		} else {
			font.drawChars(gc, Font.getStringChars(str), 0, l, x + gc.translateX, y + gc.translateY);
		}
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param str
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param len
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void drawSubstring(GraphicsContext gc, String str, int offset, int len, Font font, int x, int y) {
		assert gc != null && str != null && font != null;
		checkSubstringBounds(str, offset, len);
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			font.drawChars(gc, str.chars, str.offset + offset, len, x + gc.translateX, y + gc.translateY);
		} else {
			font.drawChars(gc, Font.getStringChars(str), offset, len, x + gc.translateX, y + gc.translateY);
		}
	}

	private static void checkSubstringBounds(String str, int offset, int len) {
		try {
			// check bounds, throw ArrayIndexOutOfBoundsException if wrong
			// values; str.length() can throw NPE
			ArrayTools.checkArrayBounds(str.length(), offset, len);
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new StringIndexOutOfBoundsException();
		}
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param character
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void drawChar(GraphicsContext gc, char character, Font font, int x, int y) {
		assert gc != null && font != null;
		font.drawChars(gc, new char[] { character }, 0, 1, x + gc.translateX, y + gc.translateY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param renderableString
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void drawRenderableString(GraphicsContext gc, RenderableString renderableString, int x, int y) {
		assert gc != null && renderableString != null;
		String string = renderableString.string;
		char[] chars;
		int offset;
		int length;
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			chars = string.chars;
			offset = string.offset;
			length = string.length;
		} else {
			chars = Font.getStringChars(string);
			offset = 0;
			length = chars.length;
		}
		PainterNatives.drawRenderableString(gc.getSNIContext(), chars, offset, length,
				renderableString.font.getSNIContext(), renderableString.width, renderableString.sniContext,
				x + gc.translateX, y + gc.translateY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public static void drawImage(GraphicsContext gc, Image image, int x, int y) {
		assert gc != null && image != null;
		drawRegion(gc, image.getSNIContext(), 0, 0, image.getWidth(), image.getHeight(), x, y, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawImage(GraphicsContext gc, Image image, int x, int y, int alpha) {
		assert gc != null && image != null;
		drawRegion(gc, image.getSNIContext(), 0, 0, image.getWidth(), image.getHeight(), x, y, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param xSrc
	 *            microUI-API
	 * @param ySrc
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param xDest
	 *            microUI-API
	 * @param yDest
	 *            microUI-API
	 */
	public static void drawImageRegion(GraphicsContext gc, Image image, int xSrc, int ySrc, int width, int height,
			int xDest, int yDest) {
		assert gc != null && image != null;
		drawRegion(gc, image.getSNIContext(), xSrc, ySrc, width, height, xDest, yDest, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param xSrc
	 *            microUI-API
	 * @param ySrc
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param xDest
	 *            microUI-API
	 * @param yDest
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawImageRegion(GraphicsContext gc, Image image, int xSrc, int ySrc, int width, int height,
			int xDest, int yDest, int alpha) {
		assert gc != null && image != null;
		drawRegion(gc, image.getSNIContext(), xSrc, ySrc, width, height, xDest, yDest, alpha);
	}

	private static void drawRegion(GraphicsContext gc, byte[] srcSd, int xSrc, int ySrc, int width, int height,
			int xDest, int yDest, int alpha) {

		// compute destination area
		xDest += gc.translateX;
		yDest += gc.translateY;

		PainterNatives.drawImage(gc.getSNIContext(), srcSd, xSrc, ySrc, width, height, xDest, yDest, alpha);
	}
}
