/*
 * Java
 *
 * Copyright 2016-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.annotation.Nullable;
import ej.kf.DeadFeatureException;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.microui.MicroUIFactory;

/**
 * Special event for switching the display related by the {@link DisplayPump}.
 */
public class KFSetCurrentDisplayRunnable extends WaitForEventRunnable {

	private final KFDisplayPump displayPump;
	@Nullable
	private Display newDisplay;

	/**
	 * @param displayPump
	 *            the pump
	 * @param newDisplay
	 *            the new display to switch
	 */
	public KFSetCurrentDisplayRunnable(KFDisplayPump displayPump, Display newDisplay) {
		this.displayPump = displayPump;
		this.newDisplay = newDisplay;
	}

	@Override
	protected synchronized void notifyWaitingThread() {
		// There may be multiple waiting threads on the same event
		// See comments in KFDisplayPump.switchDisplay()
		notifyAll();
	}

	@Override
	public void executeEvent() {
		KFDisplayPump pump = this.displayPump;
		Display previousDisplay = pump.display;

		// hide previous Displayable
		try {
			previousDisplay.replaceDisplayable(null);
		} catch (DeadFeatureException e) {
			// previous display being killed - skip
		}

		pump.setCurrentDisplay(this.newDisplay);

		MicroUIFactory.getInstance().resetEventHandlers(this.newDisplay);
	}

	@Override
	protected void cancelEvent() {
		// remove the reference to the display
		this.newDisplay = null;
	}

	public boolean isNewDisplayOwnedBy(Module m) {
		return Kernel.getOwner(this.newDisplay) == m;
	}

	public boolean isSameNewDisplay(KFSetCurrentDisplayRunnable other) {
		return this.newDisplay == other.newDisplay;
	}
}
