/*
 * Java
 *
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.annotation.Nullable;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.microui.MicroUI;
import ej.microui.MicroUIFactory;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventGeneratorsPool;
import ej.microui.event.EventHandler;

public class KFMicroUIFactory extends MicroUIFactory {

	static {
		Kernel.addFeatureStateListener(new DisplayFeatureStateListener());
	}

	/**
	 * Kernel lazy initialization method for {@link MicroUIFactory#INSTANCE}
	 */
	public static MicroUIFactory createFactory() {
		if (Kernel.isInKernelMode()) {
			return new KFMicroUIFactory();
		} else {
			Kernel.enter();
			return MicroUIFactory.getInstance();
		}
	}

	private static Display getKernelDisplay() {
		Kernel.enter();
		return Display.INSTANCE;
	}

	public static Display createDisplay() {
		if (Kernel.isInKernelMode()) {
			return Display.createDisplay();
		} else {
			// Feature context.
			// Kernel Displays have been initialized
			// Create Feature owned displays
			return new KFDisplayFeature(getKernelDisplay());
		}
	}

	@Override
	public DisplayPump newDisplayPump(Display d) {
		return new KFDisplayPump(d);
	}

	@Override
	public void setEventHandler(EventGenerator gen, @Nullable EventHandler handler) {
		// check permission only when modifying a system event generator
		if (Kernel.getOwner(gen) == Kernel.getInstance()) {
			gen.checkEventHandlerPermission();
			Kernel.enter(); // allow to connect K->F link (will be removed when Feature is stopped - see
			// resetEventGeneratorsOwnedBy())
			if (handler != null && Kernel.getOwner(handler) == Kernel.getInstance()) {
				// a Kernel handler is set as default handler if it is the first
				// registered handler (System MicroUI)
				gen.setDefaultEventHandler0(handler);
			}
		}
		gen.setEventHandler0(handler);
	}

	/**
	 * KF EventHandler semantic:
	 * <ul>
	 * <li>A Feature can register its own event handler (subject to permission checks)</li>
	 * <li>When Display.show() is called, for each event generator related to the display that is connected to a handler
	 * not accessible to the display, the system tries to set back the default event handler. This operation is also
	 * subject to permission checks (silently ignored).</li>
	 * </ul>
	 */
	@Override
	public void resetEventHandlers(Display display) {
		Kernel.enter();
		Module displayOwner = Kernel.getOwner(display);
		for (final EventGenerator gen : EventGeneratorsPool.SystemPool.get()) {
			if (gen == null) {
				// a generator has been removed from system pool
				continue;
			}
			final EventHandler defaultEventHandler = gen.getDefaultEventHandler();
			EventHandler currentEventHandler = gen.getEventHandler();
			if (currentEventHandler != defaultEventHandler) {
				// the current event handler is not the default one (the
				// display event serializer)
				if (currentEventHandler == null || Kernel.getOwner(currentEventHandler) != displayOwner) {
					// No event handler or the current event handler is not
					// owned by the same context => try to reset to default event handler
					Kernel.runUnderContext(displayOwner, new Runnable() {

						@Override
						public void run() {
							try {
								gen.setEventHandler(defaultEventHandler);
							} catch (SecurityException e) {
								// kernel does not allow the current context
								// to connect this handler
								// => silently ignore
							}
						}
					});
				}
			}
		}
	}

	@Override
	public void start(MicroUI microUI) {
		Kernel.enter();
		super.start(microUI);
	}

	@Override
	public void stop(MicroUI microUI) {
		Kernel.enter();
		super.stop(microUI);
	}
}

class DisplayFeatureStateListener implements FeatureStateListener {

	@Override
	public synchronized void stateChanged(Feature feature, State previousState) {
		if (feature.getState() == State.STOPPED) {
			resetDisplayOwnedBy(feature);
			resetEventGeneratorsOwnedBy(feature);
		}
	}

	private void resetEventGeneratorsOwnedBy(Feature feature) {
		Kernel.enter();
		EventGenerator[] generators = EventGeneratorsPool.SystemPool.get();
		for (EventGenerator generator : generators) {
			if (generator != null && generator.getEventHandler() != null
					&& Kernel.getOwner(generator.getEventHandler()) == feature) {
				generator.setEventHandler0(generator.getDefaultEventHandler());
			}
		}
	}

	/**
	 * When the Feature currently owns a Display, it is reset to the associated Kernel one.
	 */
	private void resetDisplayOwnedBy(Feature feature) {
		Kernel.enter();
		if (MicroUI.isStarted()) {
			Display d = Display.getDisplay();
			((KFDisplayPump) d.getEventSerializer()).resetDisplayIfOwnedBy(feature, d);
		}
		// else: the firmware has not started MicroUI (prevent useless exception)
	}
}
