/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemPump;
import com.is2t.microui.watchdog.KFWatchdog;

import ej.bon.Constants;
import ej.kf.DeadFeatureException;
import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.microui.MicroUIProperties;
import ej.microui.SystemMicroUI;

public class KFDisplayPump extends DisplayPump {

	/**
	 * Event to switch display when starting/stopping a feature.
	 */
	/* default */ static final int EVENT_DISPLAY_SWITCH_DISPLAY = EVENT_DISPLAY | 0x07;

	/* default */ Display display;

	/**
	 * Watchdog started in KF display pump when pumping. If pump event execution is taking too much time, it kills the
	 * feature that generate the event.
	 */
	private final KFWatchdog pumpWatchdog = new KFWatchdog();

	public KFDisplayPump(Display display) {
		super();
		this.display = display;
	}

	@Override
	protected Display getDisplay() {
		return this.display;
	}

	@Override
	public boolean handleEvent(int event) {
		// Adding a new event is done in Kernel mode.
		Kernel.enter();
		return super.handleEvent(event);
	}

	@Override
	public boolean isCurrentDisplay(Display d) {
		Kernel.enter();
		return this.display == d;
	}

	@Override
	protected void executeEvent(int event, int eventType) throws InterruptedException {
		this.pumpWatchdog.start(Constants.getInt(MicroUIProperties.PUMP_WATCHDOG_MAX_DELAY),
				Kernel.getOwner(this.display));

		if (eventType == EVENT_DISPLAY_SWITCH_DISPLAY) {
			((Runnable) (getEventData(event))).run();
		} else {
			super.executeEvent(event, eventType);
		}
	}

	@Override
	protected void executeDone() {
		this.pumpWatchdog.close();
		super.executeDone();
	}

	/**
	 * Switch the DisplayPump owner. This action is serialized by DisplayPump thread. The display pump queue is cleared.
	 * This function waits until the current event is terminated, so that when this method returns, the queue is empty,
	 * the last event has been fully executed and the current display has been set.
	 *
	 * @return <code>true</code> if the pump owns the display, <code>false</code> otherwise
	 */
	@Override
	public boolean switchDisplay(Display newDisplay) {
		// here, permission is granted => add event to set the current display
		Kernel.enter();

		// true if the display switch must be executed now in the current thread, false
		// otherwise.

		KFSetCurrentDisplayRunnable newEventData;
		boolean isDPThread = Thread.currentThread() == getThread();
		boolean executeNow = isDPThread;

		synchronized (this) { // sync on runnable buffer
			// create new Kernel event
			newEventData = new KFSetCurrentDisplayRunnable(this, newDisplay);

			int oldEvent = NSystemPump.getKernelEvent(SystemMicroUI.getTypeMask(EVENT_DISPLAY_SWITCH_DISPLAY));
			if (oldEvent != 0) {
				// there is already a Kernel event at index 0
				KFSetCurrentDisplayRunnable oldEventData = (KFSetCurrentDisplayRunnable) readEventData(oldEvent);
				if (oldEventData.isSameNewDisplay(newEventData)) {
					// there is already a pending request for the same display
					// => reuse the old event.
					// In case of !executeNow, there will be 2 threads waiting for the same
					// event execution. See the 'notifyAll()' in KFSetCurrentDisplayRunnable.notifyWaitingThread()
					newEventData = oldEventData;
				} else {
					// this event is not requesting the same display
					// cancel the operation to unlock the potential waiting thread
					oldEventData.cancel();
					// => destroy the old event data and replace the Kernel event in the queue by the new one
					getEventData(oldEvent);
					int newEvent = SystemMicroUI.buildEvent(EVENT_DISPLAY_SWITCH_DISPLAY, storeEventData(newEventData));
					NSystemPump.replaceKernelEvent(newEvent);
				}
			} else {
				if (!executeNow) {
					int newEvent = SystemMicroUI.buildEvent(EVENT_DISPLAY_SWITCH_DISPLAY, storeEventData(newEventData));
					NSystemPump.addKernelEvent(newEvent);
				}
			}
		}

		if (executeNow) {
			// Here, the thread is the DiplayPump
			// Execute the display switch now.
			// We are no more synchronized on a Kernel object.
			// If the event is already in the queue, it will be dequeued later by the UI pump but will not be
			// executed because will no more be in the PENDING state.
			newEventData.run();
			return true;
		} else {
			// Here, the thread is not the DiplayPump
			// Wait for execution of CLEAR event
			return newEventData.waitForExecution();
		}
	}

	/**
	 * @return <code>true</code> if the pump owns the display, <code>false</code> otherwise
	 * @throws SecurityException
	 *             if access is not permitted based on the current security policy.
	 */
	private boolean requestPhysicalDisplayAccess(Module module) {
		final Display display = this.display;
		Module displayOwner = Kernel.getOwner(display);
		if (displayOwner != module) {
			// event owned by a Feature: only added if it is the current owner of the Pump
			// => need to request the Display access

			final boolean[] result = new boolean[1];
			Kernel.runUnderContext(module, new Runnable() {
				@Override
				public void run() {
					result[0] = Display.getDisplay().requestPhysicalDisplayAccess();
				}
			});
			return result[0];
		} else {
			return true;
		}
	}

	@Override
	public void createAndHandleEvent(int event) {
		// Adding a new event is done in Kernel mode.
		// Caller has not called Kernel.enter() for sure
		Module m = Kernel.getContextOwner();
		Kernel.enter();
		if (requestPhysicalDisplayAccess(m)) {
			super.createAndHandleEvent(event);
		} else {
			// the display has been switched to an other context in the meantime
			// this is common if the Kernel does not provide a SecurityManager or if the SecurityManager allows multiple
			// contexts to take the Display at a time.
			// => silently skip the event
		}
	}

	@Override
	public void createAndHandleEvent(int event, Object eventData) {
		// Adding a new event is done in Kernel mode.
		Kernel.enter();
		if (requestPhysicalDisplayAccess(Kernel.getOwner(eventData))) {
			super.createAndHandleEvent(event, eventData);
		} else {
			// the display has been switched to an other context in the meantime
			// this is common if the Kernel does not provide a SecurityManager or if the SecurityManager allows multiple
			// contexts to take the Display at a time.
			// => silently skip the event
		}
	}

	@Override
	protected void handleError(Throwable e) {

		/**
		 * In case pump watchdog has been triggered, pump will catch a DFE. In that case, since KF has no dead listener
		 * yet, we ignore the exception in order to prevent to add the following code in kernel:
		 *
		 * <code>
		 * Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
		 * 	public void uncaughtException(Thread t, Throwable e) {
		 * 		if (!MicroUI.isUIThread(t) || e.getClass() != DeadFeatureException.class) {
		 * 			e.printStackTrace();
		 * 		}
		 * 		// else: Feature's MicroUI pump correctly released.
		 * 	}
		 * });
		 * </code>
		 *
		 */

		try {
			if (!(e instanceof DeadFeatureException)) {
				super.handleError(e);
			}
		} catch (Throwable ex) {
			// An application error occurred but error handling failed:
			// Possible reasons:
			// - the owner of the error (= owner of the Display) has been killed
			// - the Display has been detached (NPE when executing events in the
			// queue)
			// => silently skip
		}
	}

	/**
	 * Detach the Display if it is owned by the given Feature and attach it to default Kernel one
	 */
	public synchronized void resetDisplayIfOwnedBy(Feature feature, Display kernelDisplay) {
		// unlock potential Feature thread
		for (Object eventData : eventsData) {
			if (eventData instanceof KFSetCurrentDisplayRunnable) {
				KFSetCurrentDisplayRunnable eventDataKF = (KFSetCurrentDisplayRunnable) eventData;
				if (eventDataKF.isNewDisplayOwnedBy(feature)) {
					eventDataKF.cancel();
				}
				// at most one event of KFSetCurrentDisplayRunnable in the queue at a time
				// see switchDisplay() old event replacement
				break;
			}
		}

		// Switch the display to unlink the Feature display reference
		if (Kernel.getOwner(this.display) == feature) {
			setCurrentDisplay(kernelDisplay);
		}
	}

	/**
	 * Change the {@link Display} currently connected to the Pump and clear the internal state.
	 */
	public synchronized void setCurrentDisplay(Display d) {
		clearQueue();
		d.resetFlush();
		this.display = d;
	}

	@Override
	void callOnFlushCompleted(Runnable r) {
		if (Kernel.getOwner(this.display) == (null == r ? Kernel.getContextOwner() : Kernel.getOwner(r))) {
			// Adding a new event is done in Kernel mode.
			Kernel.enter();
			super.callOnFlushCompleted(r);
		}
		// else: caller cannot add its runnable or remove the current runnable
	}
}
