/*
 * Java
 *
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import java.util.NoSuchElementException;

import com.is2t.tools.ArrayTools;

import ej.bon.Immutables;
import ej.kf.Kernel;
import ej.microui.MicroUIProperties;

public class KFDisplayFont {

	private KFDisplayFont() {
		// not entry point
	}

	/**
	 * Kernel lazy initialization method for {@link RasterFont#Fonts}. Feature fonts are composed of Kernel fonts and
	 * additional fonts described in an immutable string.
	 */
	public static RasterFont[] initializeDisplayFonts() {
		// NOTE: When entering here, monitor on Font class has already
		// been locked (caller is Font.getFonts())
		// Subsequent call to getFonts0() in Kernel mode is done without taking the
		// monitor again
		// because a monitor is not allowed to be entered in 2 different contexts.

		if (Kernel.isInKernelMode()) {
			return RasterFont.initializeAllFonts();
		} else {

			// add Kernel fonts
			RasterFont[] fonts = (RasterFont[]) ArrayTools.add(getKernelFonts(), new RasterFont[0]);

			String userFonts;
			try {
				userFonts = (String) Immutables.get(MicroUIProperties.FONTS_LIST);
			} catch (NoSuchElementException e) {
				return fonts;
			}

			// Load Feature fonts
			return RasterFont.addFonts(userFonts, fonts);
		}
	}

	private static Font[] getKernelFonts() {
		Kernel.enter();
		return RasterFont.getFontsInternal(); // see comment in initializeDisplayFonts()
	}
}
