/*
 * Java
 *
 * Copyright 2010-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;
import com.is2t.microbsp.microui.natives.NTools;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.Log;
import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microui.MicroUIProperties;
import ej.trace.Tracer;

/**
 * microUI-API
 */
public class Image {

	/*
	 * These values are a copy of values available in {@link SImageMetadata}. There are used to inline the calls.
	 */
	private final int width;
	private final int height;

	/**
	 * Icetea object: ScreenDescriptor. null when the image is closed
	 */
	protected byte[] sd;

	/**
	 * microUI-API
	 */
	/* default */ Image() {
		this(null, 0, 0);
	}

	/**
	 * Constructor for images
	 */
	/* default */ Image(byte[] sd, int logOpen, int logType) {
		this.sd = sd;
		this.width = SImageMetadata.getWidth(sd);
		this.height = SImageMetadata.getHeight(sd);
		logCreateImageEnd(sd, logOpen, logType);
	}

	/**
	 * Check if MicroUI is running, the security manager and the given path.
	 *
	 * @param path
	 *            the path to check
	 * @param permission
	 *            a <code>ImagePermission</code> permission
	 * @return the trimmed path
	 */
	/* default */ static String checkImagePathPrerequisites(String path, String permission) {
		assert path != null;
		checkPrerequisites(permission);
		return MicroUIException.checkResourcePath(path);

	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean canGetImage(String name) {
		name = NTools.getRawImagePath(checkImagePathPrerequisites(name, ImagePermission.PERMISSION_CAN_GET));
		return canGetImage(name, false) || canGetImage(name, true);
	}

	/**
	 * microUI-API
	 *
	 * @param name
	 *            microUI-API
	 * @return microUI-API
	 */
	public static Image getImage(String name) {

		name = NTools.getRawImagePath(checkImagePathPrerequisites(name, ImagePermission.PERMISSION_GET));

		logCreateImageStart(Log.TYPE_OPEN_IMAGE_FROM_PATH);

		byte[] sd = SImageMetadata.allocate(false);

		// 1- try to load an internal raw image
		Image image = getImage(name, sd, false);
		if (image != null) {
			// done!
			return image;
		}

		// 2- try to load an external raw image
		image = getImage(name, sd, true);
		if (image != null) {
			// done!
			return image;
		}

		throw new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH, name);
	}

	private static String getImagePath(String name, boolean externalResource) {
		return externalResource ? NTools.getExternalImagePath(name) : name;
	}

	private static boolean canGetImage(String name, boolean externalResource) {
		final String fullname = getImagePath(name, externalResource);
		return NSystemDisplay.NO_ERROR == NSystemDisplay.loadImageFromPath(fullname.getBytes(), fullname.length(),
				false, externalResource, true, NSystemDisplay.DEFAULT_FORMAT, null);
	}

	private static @Nullable Image getImage(String name, byte[] sd, boolean externalResource) {

		final String fullname = getImagePath(name, externalResource);

		int errorCode = NSystemDisplay.loadImageFromPath(fullname.getBytes(), fullname.length(), false,
				externalResource, true, NSystemDisplay.DEFAULT_FORMAT, sd);

		switch (errorCode) {
		case NSystemDisplay.NO_ERROR:
			return new Image(sd, Log.TYPE_OPEN_IMAGE_FROM_PATH, Log.IMAGE_CARACTERISTICS_INTERNAL_RAW);
		case NSystemDisplay.INVALID_PATH:
			// let caller manage the error
			return null;
		default:
			// file found but native exception: throw exception
			NTools.checkError(errorCode, name);
			return null; // never here
		}
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getWidth() {
		return this.width;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getHeight() {
		return this.height;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean isTransparent() {
		return NSystemDisplay.isTransparent(getSNIContext());
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @return microUI-API
	 */
	public int readPixel(int x, int y) {
		if (x < 0 || y < 0 || x >= getWidth() || y >= getHeight()) {
			// test is performed here to throw the exception. It is better to perform this
			// test on native side but how to throw the exception on simulator ? task
			// M0092MEJAUI-1947
			throw new IllegalArgumentException();
		}
		return NSystemDisplay.readPixel(getSNIContext(), x, y);
	}

	/**
	 * microUI-API
	 *
	 * @param argbData
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param scanlength
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public void readPixels(int[] argbData, int offset, int scanlength, int x, int y, int width, int height) {
		assert argbData != null;
		GraphicsContext.getARGB(getSNIContext(), argbData, offset, scanlength, x, y, width, height);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public byte[] getSNIContext() {
		return this.sd;
	}

	/* default */ static void logCreateImageStart(int logEvent) {
		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			Log.Instance.recordEvent(Log.OPEN_IMAGE, logEvent);
		}
	}

	/* default */ final void logCreateImageEnd(byte[] sd, int logOpen, int logType) {
		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			// trace image caracteristics (how the image has been created and image data) in
			// a specific log because log "recordEventEnd" cannot log more than one data
			int ref = SImageMetadata.getImageReference(sd);
			Log.Instance.recordEvent(Log.IMAGE_CARACTERISTICS, logOpen, logType, ref, getWidth(), getHeight());
			Log.Instance.recordEventEnd(Log.OPEN_IMAGE, ref);
		}
	}

	/**
	 * Checks whether the current state of MicroUI allows the caller to perform his action and whether the given
	 * permission is allowed by the security manager.
	 *
	 * @param permission
	 *            the permission related to the image creation API
	 */
	/* default */ static void checkPrerequisites(String permission) {

		// ensure MicroUI is started
		MicroUI.checkRunning();

		// ensure that the caller can perform his action
		if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = System.getSecurityManager();
			if (sm != null) {
				sm.checkPermission(new ImagePermission(permission));
			}
		}
	}
}
