/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;
import com.is2t.tools.ArrayTools;

import ej.bon.XMath;
import ej.microui.MicroUI;
import ej.microui.MicroUIException;

/**
 * microUI-API
 */
public class GraphicsContext {

	/**
	 * microUI-API
	 */
	public static final int OPAQUE = 0xff;

	/**
	 * microUI-API
	 */
	public static final int TRANSPARENT = 0x0;

	/**
	 * microUI-API
	 */
	public static final int DRAWING_SUCCESS = 0;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_ERROR = 1 << 31;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_NOT_IMPLEMENTED = 1 << 0;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_FORBIDDEN = 1 << 1;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_OUT_OF_MEMORY = 1 << 2;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_CLIP_MODIFIED = 1 << 3;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_MISSING_CHARACTER = 1 << 4;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_LIBRARY_INCIDENT = 1 << 29;
	/**
	 * microUI-API
	 */
	public static final int DRAWING_LOG_UNKNOWN_INCIDENT = 1 << 30;

	/**
	 * Icetea object: ScreenDescriptor
	 */
	private final byte[] sd;

	/*
	 * These values are a copy of values available in {@link SImageMetadata}. There are used to inline the calls.
	 */
	private final int width;
	private final int height;

	/* default */ final Format format;

	/* default */ int translateX;
	// current translation ABSOLUTE LOCATION
	/* default */ int translateY;

	/*
	 * These values are a copy of values available in {@link SImageMetadata}. There are used to inline the calls. They
	 * MUST NOT be updated outside setClip methods.
	 */
	private int graphicalEngineClipX;
	private int graphicalEngineClipY;
	private int graphicalEngineClipWidth;
	private int graphicalEngineClipHeight;

	/* default */ GraphicsContext(byte[] sd) {
		this(sd, MicroUI.isStarted(), Format.DISPLAY);
	}

	/* default */ GraphicsContext(byte[] sd, boolean resetContext, Format format) {
		this.sd = sd;
		this.width = SImageMetadata.getWidth(sd);
		this.height = SImageMetadata.getHeight(sd);
		this.format = format;
		if (resetContext) {
			reset();
		}
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getWidth() {
		return this.width;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getHeight() {
		return this.height;
	}

	/**
	 * microUI-API
	 *
	 * @param opacityPercent
	 *            microUI-API
	 * @return microUI-API
	 */
	public static int getAlpha(int opacityPercent) {
		return XMath.limit(opacityPercent * OPAQUE / 100, 0, OPAQUE);
	}

	/**
	 * microUI-API
	 */
	public void resetTranslation() {
		this.translateX = 0;
		this.translateY = 0;
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public void setTranslation(int x, int y) {
		this.translateX = x;
		this.translateY = y;
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	public void translate(int x, int y) {
		// change attributes. It is coherent to have translateXY < 0
		this.translateX += x;
		this.translateY += y;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getTranslationX() {
		return this.translateX;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getTranslationY() {
		return this.translateY;
	}

	/**
	 * microUI-API
	 *
	 * @param rgbColor
	 *            microUI-API
	 */
	public void setColor(int rgbColor) {
		// native world requires fully opaque color (see gc->foreground_color)
		NSystemDisplay.setForegroundColor(getSNIContext(), rgbColor | SImageMetadata.ALPHA_MASK);
	}

	/**
	 * microUI-API
	 *
	 * @param rgbColor
	 *            microUI-API
	 */
	public void setBackgroundColor(int rgbColor) {
		// native world requires a 24-bit data
		NSystemDisplay.setBackgroundColor(getSNIContext(), rgbColor & ~SImageMetadata.ALPHA_MASK, true);
	}

	/**
	 * microUI-API
	 */
	public void removeBackgroundColor() {
		// change attribute in native world
		NSystemDisplay.setBackgroundColor(getSNIContext(), 0, false);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getColor() {
		return SImageMetadata.getForegroundColor(this.sd) & ~SImageMetadata.ALPHA_MASK;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getBackgroundColor() {
		return SImageMetadata.getBackgroundColor(this.sd);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean hasBackgroundColor() {
		return SImageMetadata.hasBackgroundColor(this.sd);
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public void intersectClip(int x, int y, int width, int height) {
		NSystemDisplay.intersectClip(getSNIContext(), x + this.translateX, y + this.translateY, width, height);
		updateJavaClip();
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public void setClip(int x, int y, int width, int height) {
		NSystemDisplay.setClip(getSNIContext(), x + this.translateX, y + this.translateY, width, height);
		updateJavaClip();
	}

	/**
	 * microUI-API
	 */
	public void resetClip() {
		NSystemDisplay.resetClip(getSNIContext());
		updateJavaClip();
	}

	private void updateJavaClip() {
		this.graphicalEngineClipX = SImageMetadata.getClipX1(this.sd);
		this.graphicalEngineClipY = SImageMetadata.getClipY1(this.sd);
		this.graphicalEngineClipWidth = SImageMetadata.getClipWidth(this.sd);
		this.graphicalEngineClipHeight = SImageMetadata.getClipHeight(this.sd);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getClipX() {
		// Separating the getClip operation into two methods returning
		// integers is more fast and memory efficient than one getClip() call
		// returning an object.
		return this.graphicalEngineClipX - this.translateX;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getClipY() {
		// Separating the getClip operation into two methods returning integers
		// is more performance and memory efficient than one getClip() call
		// returning an object.
		return this.graphicalEngineClipY - this.translateY;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getClipWidth() {
		return this.graphicalEngineClipWidth;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getClipHeight() {
		return this.graphicalEngineClipHeight;
	}

	/**
	 * microUI-API
	 */
	public void notifyDrawingRegion() {
		NSystemDisplay.notifyDrawingRegion(getSNIContext());
	}

	/**
	 * microUI-API
	 *
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @return microUI-API
	 */
	public int readPixel(int x, int y) {
		x += this.translateX;
		y += this.translateY;
		if (x < 0 || y < 0 || x >= this.width || y >= this.height) {
			// test is performed here to throw the exception. It is better to perform this
			// test on native side but how to throw the exception on simulator ? task
			// M0092MEJAUI-1947
			throw new IllegalArgumentException();
		}
		return NSystemDisplay.readPixel(getSNIContext(), x, y);
	}

	/**
	 * microUI-API
	 *
	 * @param argbData
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param scanlength
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public void readPixels(int[] argbData, int offset, int scanlength, int x, int y, int width, int height) {
		assert argbData != null;
		getARGB(getSNIContext(), argbData, offset, scanlength, x, y, width, height);
	}

	/* default */ static void getARGB(byte[] sd, int[] argbData, int offset, int scanlength, int x, int y, int width,
			int height) {

		// check scan bounds
		checkScanBounds(argbData.length, offset, scanlength, width, height);

		if (x < 0 || y < 0 || x + width - 1 > SImageMetadata.getWidth(sd)
				|| y + height - 1 > SImageMetadata.getHeight(sd)) {
			throw new IllegalArgumentException();
		}
		NSystemDisplay.getARGB(sd, argbData, offset, scanlength, x, y, width, height);
	}

	/**
	 * Extracted from GeometryTools.
	 *
	 * <li>(1) check if the absolute value of scanlength is less than width</li>
	 * <li>(2) check if the scan of the array do not exceed the bounds</li>
	 */
	private static void checkScanBounds(int arrayLength, int offset, int scanlength, int width, int height) {
		// (2) max (or min depending on scanlength) bound reach by the array scan
		int endBound = (height - 1) * scanlength;

		// (1) must compare with the negative absolute value of scanlength in order to
		// manage scanlength = Integer.MIN_VALUE
		// because -Integer.MIN_VALUE == Integer.MIN_VALUE
		int scanlengthNeg;
		int widthNeg = -width;

		if (scanlength > 0) {
			scanlengthNeg = -scanlength;
		} else {
			scanlengthNeg = scanlength;
			offset += endBound;
			endBound = -endBound;
		}
		endBound += width;

		// (1)
		if (widthNeg < scanlengthNeg) {
			throw new IllegalArgumentException();
		}

		// (2)
		ArrayTools.checkArrayBounds(arrayLength, offset, endBound);
	}

	/**
	 * microUI-API
	 *
	 * @param width
	 *            microUI-API
	 */
	public void enableEllipsis(int width) {
		if (width <= 0) {
			throw new IllegalArgumentException();
		}
		NSystemDisplay.setEllipsis(getSNIContext(), width);
	}

	/**
	 * microUI-API
	 */
	public void disableEllipsis() {
		NSystemDisplay.setEllipsis(this.sd, 0); // 0 means "no ellipsis"
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getEllipsisWidth() {
		return SImageMetadata.getEllipsis(this.sd);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getAndClearDrawingLogFlags() {
		return NSystemDisplay.getAndClearDrawingLogFlags(this.sd);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 * @throws MicroUIException
	 *             microUI-API
	 */
	public int checkDrawingLogFlags() {
		final int flags = getAndClearDrawingLogFlags();

		if ((flags & DRAWING_LOG_ERROR) != 0) {
			throw new MicroUIException(MicroUIException.DRAWING_ERROR, Integer.toHexString(flags));
		}

		return flags;
	}

	/**
	 * microUI-API
	 */
	public final void reset() {
		resetTranslation();
		resetClip();
		setColor(Colors.BLACK);
		removeBackgroundColor();
		disableEllipsis();
		getAndClearDrawingLogFlags();
	}

	/**
	 * not in api<br>
	 * kept for backward compatibility. Called by SystemMicroUI on startup.
	 */
	public void resetContext() {
		reset();
	}

	/**
	 * microUI-API
	 *
	 * @return metadata
	 */
	public byte[] getSNIContext() {
		byte[] sd = this.sd;
		if (sd == null) {
			// image has been closed
			throw new MicroUIException(MicroUIException.RESOURCE_CLOSED);
		}
		return sd;
	}

}
