/*
 * Java
 *
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

/**
 * Contains all the available buffer formats.
 */
public enum Format {

	/**
	 * Same format as the display.
	 * <p>
	 * It can be an alias for one of the other formats if the display uses a standard format. Otherwise, it represents
	 * the display custom format.
	 * <p>
	 * Useful to create a buffered image with the same format as the display for instance.
	 */
	DISPLAY(0x00),
	/**
	 * ARGB8888 format.
	 */
	ARGB8888(0x02),
	/**
	 * ARGB1555 format.
	 */
	ARGB1555(0x05),
	/**
	 * ARGB4444 format.
	 */
	ARGB4444(0x06),
	/**
	 * ARGB8888 pre-multiplied format (each color component is multiplied by the opacity).
	 */
	ARGB8888_PRE(0x18),
	/**
	 * ARGB4444 pre-multiplied format (each color component is multiplied by the opacity).
	 */
	ARGB4444_PRE(0x1a),
	/**
	 * ARGB1555 pre-multiplied format (each color component is multiplied by the opacity).
	 */
	ARGB1555_PRE(0x19),
	/**
	 * RGB888 format.
	 */
	RGB888(0x03),
	/**
	 * RGB565 format.
	 */
	RGB565(0x04),
	/**
	 * A8 format.
	 */
	A8(0x08),
	/**
	 * A4 format.
	 */
	A4(0x07),
	/**
	 * A2 format.
	 */
	A2(0x0b),
	/**
	 * A1 format.
	 */
	A1(0x0c),
	/**
	 * C4 format.
	 */
	C4(0x0d),
	/**
	 * C2 format.
	 */
	C2(0x0e),
	/**
	 * C1 format.
	 */
	C1(0x0f),
	/**
	 * AC44 format.
	 */
	AC44(0x10),
	/**
	 * AC22 format.
	 */
	AC22(0x11),
	/**
	 * AC11 format.
	 */
	AC11(0x12),

	/**
	 * Custom format 0.
	 */
	CUSTOM_0(0xff),
	/**
	 * Custom format 1.
	 */
	CUSTOM_1(0xfe),
	/**
	 * Custom format 2.
	 */
	CUSTOM_2(0xfd),
	/**
	 * Custom format 3.
	 */
	CUSTOM_3(0xfc),
	/**
	 * Custom format 4.
	 */
	CUSTOM_4(0xfb),
	/**
	 * Custom format 5.
	 */
	CUSTOM_5(0xfa),
	/**
	 * Custom format 6.
	 */
	CUSTOM_6(0xf9),
	/**
	 * Custom format 7.
	 */
	CUSTOM_7(0xf8);

	private final byte value;

	private Format(int value) {
		this.value = (byte) value;
	}

	/**
	 * microUI-API
	 *
	 * Returns the SNI context data of this format.
	 *
	 * @return the value.
	 */
	public byte getSNIContext() {
		return this.value;
	}
}