/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;

/**
 * This class allows to parse the native (IceTea) data of a font.
 */
public class FontData {

	/*
	 * Offsets of data filled by native side during font loading
	 */

	/**
	 * Font's sni context offset
	 */
	/* package */static final int MAIN_OFFSET = 4;

	/**
	 * external | monospace | styles
	 */
	private static final int LOAD_OFFSET_FLAGS = MAIN_OFFSET + 4;

	/**
	 * size of identifiers array
	 */
	private static final int LOAD_OFFSET_ARRAY_ID_SIZE = MAIN_OFFSET + 6;

	/**
	 * size of descriptor
	 */
	private static final int LOAD_OFFSET_ARRAY_DESC_SIZE = MAIN_OFFSET + 7; // NOSONAR available but not used in Java

	private static final int LOAD_OFFSET_HEIGHT = MAIN_OFFSET + 8;
	private static final int LOAD_OFFSET_BASELINE = MAIN_OFFSET + 10;

	/**
	 * Size of array to allocate to use a font in native side.
	 *
	 * @see SFontJavaData.iceTea
	 */
	private static final int LOAD_OFFSETS_ARRAY_SIZE = MAIN_OFFSET + 12;

	/**
	 * Font id when a font has been closed
	 */
	private static final int FONT_CLOSED_ID = -1;

	/**
	 * EJF masks and shifts
	 */
	private static final int FONT_STYLES_SHIFT = 4;
	private static final int FONT_STYLES_MASK = 0xf0;
	private static final int FONT_MASK_MONO = 0x100;
	private static final int FONT_MASK_BOLD = 0x1;
	private static final int FONT_MASK_ITALIC = 0x2;

	private FontData() {
		// private constructor
	}

	public static byte[] allocate() {
		return new byte[LOAD_OFFSETS_ARRAY_SIZE];
	}

	public static char getHeight(byte[] nativeData) {
		return Tools.getChar(nativeData, LOAD_OFFSET_HEIGHT);
	}

	public static char getBaseline(byte[] nativeData) {
		return Tools.getChar(nativeData, LOAD_OFFSET_BASELINE);
	}

	public static int[] getIdentifiers(byte[] nativeData) {
		int l = Tools.getUnsignedByte(nativeData, LOAD_OFFSET_ARRAY_ID_SIZE);
		if (l > 0) {
			int[] array = new int[l];
			RasterFont.checkFontLoadingError(NSystemDisplay.getFontIdentifiers(nativeData, array));
			return array;
		} else {
			return new int[] { FontIdentifiers.LATIN };
		}
	}

	public static boolean isMonospaced(byte[] nativeData) {
		return isFlagSet(nativeData, FONT_MASK_MONO);
	}

	public static boolean isBold(byte[] nativeData) {
		return isFlagSet(nativeData, FONT_MASK_BOLD << FONT_STYLES_SHIFT);
	}

	public static boolean isItalic(byte[] nativeData) {
		return isFlagSet(nativeData, FONT_MASK_ITALIC << FONT_STYLES_SHIFT);
	}

	private static boolean isFlagSet(byte[] nativeData, int flag) {
		return (getFlags(nativeData) & flag) == flag;
	}

	private static char getFlags(byte[] nativeData) {
		return Tools.getChar(nativeData, LOAD_OFFSET_FLAGS);
	}
}
