/*
 * Java
 *
 * Copyright 2008-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.tools.ArrayTools;

import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.MicroUIProperties;

/**
 * microUI-API
 */
public abstract class Font {

	/**
	 * @deprecated
	 */
	@Deprecated
	public static final int STYLE_PLAIN = RasterFont.STYLE_PLAIN;

	/**
	 * @deprecated
	 */
	@Deprecated
	public static final int STYLE_BOLD = RasterFont.STYLE_BOLD;

	/**
	 * @deprecated
	 */
	@Deprecated
	public static final int STYLE_ITALIC = RasterFont.STYLE_ITALIC;

	private final byte[] sniContext;

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public static Font[] getAllFonts() {
		checkPrerequisites(/* FontPermission.PERMISSION_GET_ALL */); // permission not yet in api
		return RasterFont.getAllFonts();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public static Font getDefaultFont() {
		checkPrerequisites(/* FontPermission.PERMISSION_GET_DEFAULT */); // permission not yet in api
		return RasterFont.getDefaultFont();
	}

	/**
	 * microUI-API
	 *
	 * @param path
	 *            microUI-API
	 *
	 * @return microUI-API
	 */
	public static Font getFont(String path) {
		checkPrerequisites(/* FontPermission.PERMISSION_GET */); // permission not yet in api
		return RasterFont.getFont(path);
	}

	/**
	 * microUI-API
	 *
	 * @param format
	 *            microUI-API
	 * @param fontData
	 *            microUI-API
	 */
	protected Font(Format format, byte[] fontData) {
		this(format, fontData, true);
	}

	/**
	 * Only used by {@link RasterFont}
	 */
	/* package */ Font(byte[] fontData) {
		this(Format.DISPLAY, fontData, false);
	}

	private Font(Format format, byte[] fontData, boolean customRequired) {
		if (customRequired
				&& (format.ordinal() < Format.CUSTOM_0.ordinal() || format.ordinal() > Format.CUSTOM_7.ordinal())) {
			throw new IllegalArgumentException();
		}
		int customLength = fontData.length;
		byte[] sniContext = new byte[customLength + FontData.MAIN_OFFSET];
		sniContext[0] = format.getSNIContext();
		System.arraycopy(fontData, 0, sniContext, FontData.MAIN_OFFSET, customLength);
		this.sniContext = sniContext;
	}

	/**
	 * microUI-API
	 *
	 * @param ch
	 *            microUI-API
	 * @return microUI-API
	 */
	public int charWidth(char ch) {
		return charsWidth(new char[] { ch }, 0, 1);
	}

	/**
	 * microUI-API
	 *
	 * @param str
	 *            microUI-API
	 * @return microUI-API
	 */
	public int stringWidth(String str) {
		assert str != null;
		int length = str.length(); // can throw NPE (spec)
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			return charsWidth(str.chars, str.offset, length);
		} else {
			return charsWidth(getStringChars(str), 0, length);
		}

	}

	/**
	 * microUI-API
	 *
	 * @param str
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param len
	 *            microUI-API
	 * @return microUI-API
	 */
	public int substringWidth(String str, int offset, int len) {
		assert str != null;
		try {
			// check bounds, throw ArrayIndexOutOfBoundsException if wrong
			// values; str.length() can throw NPE
			ArrayTools.checkArrayBounds(str.length(), offset, len);
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new StringIndexOutOfBoundsException();
		}

		// perform this check AFTER checkArrayBounds (spec)

		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			return charsWidth(str.chars, str.offset + offset, len);
		} else {
			return charsWidth(getStringChars(str), offset, len);
		}
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public abstract int getHeight();

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public abstract int getBaselinePosition();

	/**
	 * not in API
	 *
	 * Copy string characters into a new array. Useful when cannot use EDC internal API.
	 *
	 * @param str
	 *            the string to extract.
	 * @return a characters array.
	 */
	public static char[] getStringChars(String str) {
		int length = str.length();
		char[] chars = new char[length];
		str.getChars(0, length, chars, 0);
		return chars;
	}

	/**
	 * microUI-API
	 *
	 * @param chars
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 * @return microUI-API
	 */
	protected int charsWidth(char[] chars, int offset, int length) {
		return PainterNatives.stringWidth(chars, offset, length, getSNIContext());
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param chars
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 */
	protected void drawChars(GraphicsContext gc, char[] chars, int offset, int length, int x, int y) {
		PainterNatives.drawString(gc.getSNIContext(), chars, offset, length, getSNIContext(), x, y);
	}

	/**
	 * microUI-API
	 *
	 * @param chars
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 * @return microUI-API
	 */
	protected byte[] allocateRenderableStringSNIContext(char[] chars, int offset, int length) {
		return new byte[0];
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public int[] getIdentifiers() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean isIdentifierSupported(int identifier) {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public int getStyle() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public String getDescriptor() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean isPlain() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean isBold() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean isItalic() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @deprecated
	 */
	@Deprecated
	public boolean isMonospaced() {
		throw new UnsupportedOperationException();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public byte[] getSNIContext() {
		return this.sniContext;
	}

	/**
	 * Checks whether the current state of MicroUI allows the caller to perform his action
	 */
	/* default */ static void checkPrerequisites() {
		// ensure MicroUI is started
		MicroUI.checkRunning();
	}

	/**
	 * Checks whether the current state of MicroUI allows the caller to perform his action and whether the given
	 * permission is allowed by the security manager.
	 *
	 * @param permission
	 *            the permission related to the image creation API
	 */
	/* default */ static void checkPrerequisites(String permission) {

		// ensure MicroUI is started
		MicroUI.checkRunning();

		// ensure that the caller can perform his action
		if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
			SecurityManager sm = System.getSecurityManager();
			if (sm != null) {
				sm.checkPermission(new FontPermission(permission));
			}
		}
	}
}
