/*
 * Java
 *
 * Copyright 2010-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microui.event.EventHandler;

/**
 * microUI-API
 */
public abstract class Displayable implements EventHandler {

	private boolean pendingRequestRender;// false init

	/**
	 * microUI-API
	 */
	public void requestRender() {
		Display display = Display.getDisplay();
		if (display.isShown(this) && (!this.pendingRequestRender || !MicroUI.isUIThread())) {

			// if this boolean is set after adding the event, a schedule may occur and the
			// boolean will be set after the call to render(); this solution prevents to
			// synchronize the caller.
			this.pendingRequestRender = true;

			try {
				display.getEventSerializer().createAndHandleEvent(DisplayPump.EVENT_DISPLAY_REPAINT_DISPLAYABLE, this);
			} catch (MicroUIException e) {
				// cannot add more events in queue: reset pending request boolean
				this.pendingRequestRender = false;
				throw e;
			}
		}
	}

	/**
	 * microUI-API
	 */
	protected void onShown() {
		// does nothing by default
	}

	/**
	 * microUI-API
	 */
	protected void onHidden() {
		// does nothing by default
	}

	/* default */ void render() {
		this.pendingRequestRender = false;
		Display d = Display.getDisplay();
		if (d.isShown(this)) {
			GraphicsContext g = d.getGraphicsContext();
			// reset graphics context (color, translate...)
			g.reset();
			// paint!
			render(g);
			// flush drawings immediately
			d.flush();
		}
		// else: nothing to render
	}

	/**
	 * microUI-API
	 *
	 * @param g
	 *            microUI-API
	 */
	protected abstract void render(GraphicsContext g);

}
