/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;

import ej.bon.Constants;
import ej.microui.Log;
import ej.microui.MicroUIPump;
import ej.microui.event.Event;
import ej.microui.event.EventHandler;
import ej.trace.Tracer;

/**
 * Extension of pump to manage display events.
 */
public class DisplayPump extends MicroUIPump {

	/**
	 * Internal {@link Event} identifiers to add {@link Display} events. Events are known by the graphical engine.
	 * <p>
	 * Events 0x08 to 0x0f are reserved to display events
	 * <p>
	 * Events 0x0A and 0x0C are not used.
	 * <p>
	 * Next updates comment: if the number of events is higher than 8 (0x08 to 0x0f), another implementation is to use
	 * only the event 0x08 and to encode sub display event in the next byte (because the data (displayable index in
	 * events array) is never higher than 0xffff).
	 * <p>
	 * See Update subclass(es) !
	 */
	/* default */ static final int EVENT_DISPLAY = 0x08;
	/* default */ static final int EVENT_DISPLAY_SHOW_DISPLAYABLE = EVENT_DISPLAY | 0x00;
	/* default */ static final int EVENT_DISPLAY_HIDE_DISPLAYABLE = EVENT_DISPLAY | 0x01;
	/* default */ static final int EVENT_DISPLAY_FLUSH = EVENT_DISPLAY | 0x03;
	/* default */ static final int EVENT_DISPLAY_REPAINT_DISPLAYABLE = EVENT_DISPLAY | 0x05;
	/* default */ static final int EVENT_DISPLAY_REPAINT_CURRENT_DISPLAYABLE = EVENT_DISPLAY | 0x06;

	/**
	 * Overrided by KF
	 *
	 * @return the pump's display
	 */
	protected Display getDisplay() {
		return Display.getDisplay();
	}

	/**
	 * Manages display events before calling user event handler if required
	 */
	@Override
	protected void executeEvent(int event, int eventType) throws InterruptedException {

		Display d = getDisplay();

		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			if (isDisplayEvent(eventType)) {
				this.log = Log.EXECUTE_DISPLAY_EVENT;
				this.logData = eventType;
				Log.Instance.recordEvent(this.log, this.logData, event);
			}
		}

		switch (eventType) {

		case EVENT_DISPLAY_FLUSH:
			d.flush();
			break;
		case EVENT_DISPLAY_SHOW_DISPLAYABLE:
			d.executeEventOnShow((Displayable) getEventData(event));
			break;
		case EVENT_DISPLAY_HIDE_DISPLAYABLE:
			d.executeEventOnHide((Displayable) getEventData(event));
			break;
		case EVENT_DISPLAY_REPAINT_DISPLAYABLE:
			d.executeEventOnPaint((Displayable) getEventData(event));
			break;
		case EVENT_DISPLAY_REPAINT_CURRENT_DISPLAYABLE:
			d.executeEventOnPaint();
			break;

		default:
			// User events
			super.executeEvent(event, eventType);
			break;
		}
	}

	@Override
	protected EventHandler getPumpHandler() {
		// the pump's event handler is the current displayable
		return getDisplay().getDisplayable();
	}

	@Override
	protected void handleError(Throwable e) {
		// catch any user errors
		// redirected to the Display in order to select the current event handler
		getDisplay().errorLog(e);
	}

	private boolean isDisplayEvent(int eventType) {
		return (eventType & EVENT_DISPLAY) == EVENT_DISPLAY;
	}

	/**
	 * See {@link KFDisplayPump}
	 *
	 * @param display
	 *            the display instance to check.
	 *
	 * @return true if the given Display is the current pump {@link Display}.
	 */
	/* default */ boolean isCurrentDisplay(Display display) {
		// NOP
		return true;
	}

	/**
	 * See {@link KFDisplayPump}
	 *
	 * @param display
	 *            the display instance to check.
	 * @return <code>true</code> if the pump owns the display, <code>false</code> otherwise
	 */
	/* default */ boolean switchDisplay(Display display) {
		// NOP
		return true;
	}

	/**
	 * Creates a call serially event and give it to native side. Native side can hold only one event. If there is
	 * already an event, native returns the old event. This old event must be read to remove the associated object (call
	 * serially event object) from objects queue.
	 *
	 * @param r
	 *            null allowed internally for {@link Display#cancelCallOnFlushCompleted()}
	 */
	/* default */ synchronized void callOnFlushCompleted(Runnable r) {

		// create the event "end of flush": it is a call serially actually!
		int event = r != null ? createEvent(MicroUIPump.CALLSERIALLY, r) : NSystemDisplay.NO_END_OF_FLUSH_EVENT;

		// give it to native side without adding it to the pump queue
		int oldEvent = NSystemDisplay.addEndOfFlushEvent(event);

		if (NSystemDisplay.NO_END_OF_FLUSH_EVENT != oldEvent) {
			// there was a "end of flush" event; this event has been
			// replaced by new one. Have to remove associated runnable
			// in objects list

			// this call gets object and removes the object from list
			getEventData(oldEvent);
		}
		// else: there was no "end of flush" event in queue
	}

	@Override
	public synchronized void clearQueue() {
		super.clearQueue();
		// reset flush state
		NSystemDisplay.addEndOfFlushEvent(NSystemDisplay.NO_END_OF_FLUSH_EVENT);
	}
}
