/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;

import ej.bon.Constants;
import ej.bon.Immortals;
import ej.microui.MicroUIFactory;
import ej.microui.MicroUIProperties;

public class DisplayFactory {

	private DisplayFactory() {
		// not the entry point
	}

	/**
	 * Creates a MicroUI display after native world initialization. Called by
	 * SystemMicroUIGenerator.
	 *
	 * @return a MicroUI Display
	 */
	public static Display newDisplay() {

		// screen descriptor of display is immortal in order to
		// keep it in display system
		byte[] sd = SImageMetadata.allocate(true);
		sd = Immortals.setImmortal(sd); // knwon by native world

		// initialize native lcd and get an unique graphics context id
		String internalString = "MicroUI"; // see simulator //$NON-NLS-1$
		char[] internStringsArray;
		if (Constants.getBoolean(MicroUIProperties.EDC_INTERNAL)) {
			internStringsArray = internalString.chars;
		} else {
			internStringsArray = Font.getStringChars(internalString);
		}
		switch (NSystemDisplay.initAndGetNextGraphicsContextId(sd, sd.length, internStringsArray)) {
		case NSystemDisplay.NO_ERROR:
			break;
		case NSystemDisplay.OUT_OF_MEMORY:
			// sd byte array is too small (MicroUI and display system are not synchronized)
			throw new OutOfMemoryError();
		default:
		case NSystemDisplay.NOK:
			// display system already initialized, driver issue etc.
			throw new RuntimeException();
		}

		MicroUIFactory factory = MicroUIFactory.getInstance();
		GraphicsContext graphics = new GraphicsContext(sd);
		Display display = new Display(graphics);

		DisplayPump pump = factory.createPump(display);
		display.setPump(pump);

		Display.INSTANCE = display;

		return display;
	}
}
