/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.bon.XMath;
import ej.microui.MicroUI;
import ej.microui.MicroUIException;
import ej.microui.MicroUIProperties;
import ej.microui.MicroUIPump;
import ej.microui.event.EventHandler;

/**
 * microUI-API
 */
public class Display {

	private static final int CONTRAST_MAX = 100;
	private static final int BACKLIGHT_MAX = 100;

	private final GraphicsContext graphics;

	/* default */ static Display INSTANCE = createDisplay();
	private @Nullable Displayable currentDisplayable;
	private DisplayPump pump;
	private boolean pendingFlush; // true if there is already a flush event in the FIFO, simplify other libraries
	// like MWT

	/* default */ Display(GraphicsContext gc) {
		// there is no current displayable
		this.graphics = gc;
		this.currentDisplayable = null;
	}

	/* default */ static Display createDisplay() {
		// no display by default: set by system microui gen (for firmware and kernel)
		// for feature: this method (via INSTANCE field) is overrided (see
		// kernel.intern)
		return null;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public EventHandler getEventHandler() {
		return this.pump;
	}

	/* default */ DisplayPump getEventSerializer() {
		return this.pump;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getHeight() {
		return this.graphics.getHeight();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getWidth() {
		return this.graphics.getWidth();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getPixelDepth() {
		// this method is rarely called: that's why we call a native to not duplicate
		// the 'bpp' field in java and in native world
		return NSystemDisplay.getNbBitsPerPixel();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean isColor() {
		// this method is rarely called: that's why we call a native to not duplicate
		// the 'color' field in java and in native world
		return NSystemDisplay.isColor();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getNumberOfColors() {
		// this method is rarely called: that's why we call a native to not duplicate
		// the 'nbColors' field in java and in native world
		return NSystemDisplay.getNumberOfColors();
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean isDoubleBuffered() {
		return NSystemDisplay.isDoubleBuffered();
	}

	/**
	 * microUI-API
	 *
	 * @param color
	 *            microUI-API
	 * @return microUI-API
	 */
	public int getDisplayColor(int color) {
		return NSystemDisplay.getDisplayColor(color & ~SImageMetadata.ALPHA_MASK) & ~SImageMetadata.ALPHA_MASK;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public boolean hasBacklight() {
		// this method is rarely called: that's why we call a native to not duplicate
		// the 'backlight' field in java and in native world
		return NSystemDisplay.hasBacklight();
	}

	/**
	 * microUI-API
	 *
	 * @param contrast
	 *            microUI-API
	 */
	public void setContrast(int contrast) {
		contrast = XMath.limit(contrast, 0, CONTRAST_MAX);
		NSystemDisplay.setContrast(contrast);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getContrast() {
		return XMath.limit(NSystemDisplay.getContrast(), 0, CONTRAST_MAX);
	}

	/**
	 * microUI-API
	 *
	 * @param backlight
	 *            microUI-API
	 */
	public void setBacklight(int backlight) {
		backlight = XMath.limit(backlight, 0, BACKLIGHT_MAX);
		NSystemDisplay.setBacklight(backlight);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getBacklight() {
		return XMath.limit(NSystemDisplay.getBacklight(), 0, BACKLIGHT_MAX);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public static Display getDisplay() {
		MicroUI.checkRunning();
		Display display = INSTANCE;
		if (display == null) {
			// API is not nullable: spec asks to throw an exception
			throw new MicroUIException(MicroUIException.NO_DISPLAY);
		}
		return display;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public @Nullable Displayable getDisplayable() {
		return this.currentDisplayable;
	}

	/**
	 * microUI-API
	 *
	 * @param displayable
	 *            microUI-API
	 */
	public void requestShow(Displayable displayable) {
		assert displayable != null;
		getEventSerializer().createAndHandleEvent(DisplayPump.EVENT_DISPLAY_SHOW_DISPLAYABLE, displayable);
	}

	/**
	 * microUI-API
	 *
	 * @param displayable
	 *            microUI-API
	 */
	public void requestHide(Displayable displayable) {
		assert displayable != null;
		getEventSerializer().createAndHandleEvent(DisplayPump.EVENT_DISPLAY_HIDE_DISPLAYABLE, displayable);
	}

	/**
	 * microUI-API
	 */
	public void requestRender() {
		getEventSerializer().createAndHandleEvent(DisplayPump.EVENT_DISPLAY_REPAINT_CURRENT_DISPLAYABLE);
	}

	/**
	 * Called by SystemMicroUIGenerator before stating UI thread ({@link MicroUIPump#start()}). This priority is
	 * deprecated since task M0092MEJAUI-2135. By consequence this method does nothing.
	 *
	 * @param priority
	 *            the pump thread priority
	 * @deprecated see task M0092MEJAUI-2135
	 */
	@Deprecated
	public void setPriority(int priority) {
		// does nothing
	}

	/**
	 * Called serially by {@link DisplayPump#execute(int)}.
	 */
	/* default */ void executeEventOnShow(Displayable displayable) {
		Displayable current = this.currentDisplayable;
		if (current != displayable) {
			// spec: show new displayable, repaint it and flush it
			replaceDisplayable(displayable);
			displayable.render();
			flush();
		}
		// else: nothing to do
	}

	/**
	 * Called serially by {@link DisplayPump#execute(int)}.
	 */
	/* default */ void executeEventOnHide(Displayable displayable) {
		Displayable current = this.currentDisplayable;
		if (current == displayable) {
			replaceDisplayable(null);
		}
		// else: nothing to do
	}

	/**
	 * Called serially by {@link DisplayPump#execute(int)}.
	 */
	/* default */ void executeEventOnPaint(Displayable displayable) {
		Displayable current = this.currentDisplayable;
		if (current == displayable) {
			// spec: paint displayable and flush it
			displayable.render();
			flush();
		}
		// else: nothing to do
	}

	/**
	 * Called serially by {@link DisplayPump#execute(int)}.
	 */
	/* default */ void executeEventOnPaint() {
		Displayable current = this.currentDisplayable;
		if (current != null) {
			// spec: paint displayable and flush it
			current.render();
			flush();
		}
		// else: nothing to do
	}

	/* default */ void replaceDisplayable(@Nullable Displayable newDisplayable) {
		Displayable current = this.currentDisplayable;
		if (current != null) {
			current.onHidden();
		}
		if (newDisplayable != null) {
			this.currentDisplayable = newDisplayable;
			newDisplayable.onShown();
		} else {
			this.currentDisplayable = null;
		}
	}

	/**
	 * microUI-API
	 */
	public void requestFlush() {
		if (!this.pendingFlush) {
			// add a flush event in the loop
			try {
				// update the boolean before adding the event in case of event is executed
				// before exiting this method (and reset the boolean, task M0092MEJAUI-2354)
				this.pendingFlush = true;
				this.pump.createAndHandleEvent(DisplayPump.EVENT_DISPLAY_FLUSH);
			} catch (MicroUIException e) {
				// the event has not been added for any reason: reset the boolean (task
				// M0092MEJAUI-1958)
				this.pendingFlush = false;
				throw e;
			}
		}
		// else: no need to ask/perform a flush because a flush is pending
	}

	/**
	 * microUI-API
	 *
	 * @throws MicroUIException
	 *             if the error flag was set by a previous drawing operation
	 */
	public void flush() {
		this.pendingFlush = false;
		if (Constants.getBoolean(MicroUIProperties.CHECK_DRAWING_ERRORS_ON_FLUSH)) {
			this.graphics.checkDrawingLogFlags();
		}
		NSystemDisplay.flushScreen(this.graphics.getSNIContext());
	}

	/**
	 * not in api
	 *
	 * Used by the MicroUI testsuite.
	 *
	 * @return the internal boolean {@link #pendingFlush}.
	 */
	/* default */ boolean getPendingFlush() {
		return this.pendingFlush;
	}

	/**
	 * microUI-API
	 */
	public void waitFlushCompleted() {
		errorIfDisplayThread();
		NSystemDisplay.waitFlush();
	}

	/**
	 * microUI-API
	 *
	 * @param r
	 *            microUI-API
	 */
	public void callOnFlushCompleted(Runnable r) {
		assert r != null;
		getEventSerializer().callOnFlushCompleted(r);
	}

	/**
	 * microUI-API
	 */
	public void cancelCallOnFlushCompleted() {
		getEventSerializer().callOnFlushCompleted(null);
	}

	/**
	 * Throws a dead lock exception if thread is display pump.
	 */
	private void errorIfDisplayThread() {
		if (MicroUI.isUIThread()) {
			throw new MicroUIException(MicroUIException.DISPLAY_DEADLOCK);
		}
	}

	/**
	 * microUI-API
	 *
	 * @param displayable
	 *            microUI-API
	 * @return microUI-API
	 */
	public boolean isShown(Displayable displayable) {
		assert displayable != null;
		return this.pump.isCurrentDisplay(this) && (this.currentDisplayable == displayable);
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public GraphicsContext getGraphicsContext() {
		return this.graphics;
	}

	/* default */ final void setPump(DisplayPump pump) {
		this.pump = pump;
	}

	/**
	 * @return <code>true</code> if the pump owns the display, <code>false</code> otherwise
	 * @throws SecurityException
	 *             if access is not permitted based on the current security policy.
	 */
	/* default */ boolean requestPhysicalDisplayAccess() {
		if (this.pump.isCurrentDisplay(this)) {
			return true;
		}

		if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
			// not owner of the display => ask for the permission
			SecurityManager sm = System.getSecurityManager();
			if (sm != null) {
				sm.checkPermission(new DisplayPermission());
			}
		}

		return this.pump.switchDisplay(this);
	}

	/* default */ void resetFlush() {
		this.pendingFlush = false;
	}

	/**
	 * Log errors related to actions on the given {@link Display}. By default, this method redirects to
	 * {@link MicroUI#errorLog(Throwable)}.
	 *
	 * @param e
	 *            the error to log
	 */
	/* default */ void errorLog(Throwable e) {
		// Internal note:
		// Subclass DisplayFeature assumes this method is called on the Display
		// => the context is switched to the display so that error is logged to the
		// UncaughtExceptionHandler of the current context
		MicroUI.MicroUI.errorLog(e);
	}
}
