
/*
 *	Java
 *
 *	Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 *	This library is provided in source code for use, modification and test, subject to license terms.
 *	Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */

package ej.microui.display;

import com.is2t.microbsp.microui.natives.NSystemDisplay;
import com.is2t.microbsp.microui.natives.NTools;

import ej.microui.Log;
import ej.microui.MicroUIException;

/**
 * microUI-API
 */
public class BufferedImage extends ResourceImage {

	private final GraphicsContext gc;

	/**
	 * microUI-API
	 *
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 */
	public BufferedImage(int width, int height) {
		this(width, height, Format.DISPLAY, ImagePermission.PERMISSION_BUFFEREDIMAGE);
	}

	/**
	 * microUI-API
	 *
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param format
	 *            microUI-API
	 */
	public BufferedImage(int width, int height, Format format) {
		this(width, height, format, ImagePermission.PERMISSION_BUFFEREDIMAGE_FORMAT);
	}

	private BufferedImage(int width, int height, Format format, String permission) {
		super(allocateSNIContext(width, height, format, permission), Log.TYPE_OPEN_IMAGE_MUTABLE,
				Log.IMAGE_CARACTERISTICS_NEW_IMAGE);
		this.gc = new MutableImageGraphicsContext(this, width, height, format);
	}

	private static byte[] allocateSNIContext(int width, int height, Format format, String permission) {
		Image.checkPrerequisites(permission);
		Image.logCreateImageStart(Log.TYPE_OPEN_IMAGE_MUTABLE);

		if (width > Character.MAX_VALUE || height > Character.MAX_VALUE || width <= 0 || height <= 0) {
			// Exceed MicroUI header file limit (size is on a char) || size is invalid
			throw new IllegalArgumentException();
		}

		// try to open the offscreen. A positive returned value is the offscreen id.
		byte[] sd = SImageMetadata.allocate(true);
		NTools.checkError(NSystemDisplay.openOffScreen(width, height, sd, format.getSNIContext()), null);
		return sd;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public GraphicsContext getGraphicsContext() {
		if (isClosed()) {
			throw new MicroUIException(MicroUIException.RESOURCE_CLOSED);
		}
		// if a image.close() is called here, gc is returned but the "close" error will
		// occur later
		return this.gc;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public Format getFormat() {
		return this.gc.format;
	}

}
