/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import ej.microui.event.generator.GenericEventGenerator;

/**
 * microUI-API
 */
public abstract class SystemMicroUI extends MicroUI {

	private static final int TYPE_MASK = 0xFF;
	private static final int TYPE_SHIFT = 24;

	/**
	 * Create an internal MicroUI implementation
	 */
	public SystemMicroUI() {
		super(false);
	}

	@Override
	protected void destroy() {
		// nothing to do
	}

	@Override
	protected void systemStart() {
		// nothing to do
	}

	@Override
	protected void systemStop() {
		// nothing to do
	}

	@Override
	protected void systemClose() {
		// nothing to do
	}

	/**
	 * Called by SystemMicroUIImpl (generated thanks SystemMicroUIGenerator)
	 *
	 * @param clazz
	 *            the generic event generator class to instanciate
	 * @return the generic event generator
	 */
	/* default */ GenericEventGenerator newGenericEventGenerator(String clazz) {
		try {
			return (GenericEventGenerator) (Class.forName(clazz).newInstance());
		} catch (Throwable ex) {
			// the class specified in VEE Port configuration file microui.xml is not in the application classpath

			// generate a warning message
			MicroUI.warning(new MicroUIException(MicroUIException.GENERIC_EVENT_GENERATOR_INVALID_CLASSNAME, clazz));

			// return a fake event generator (see InvalidEventGenerator)
			return new InvalidEventGenerator();
		}
	}

	/**
	 * Builds an event from a given type and data.
	 * <p>
	 * Type is a value between 0x0 and 0xff and data a value between 0x0 and 0xffffff (no runtime check).
	 *
	 * @param type
	 *            the type of the event to build
	 * @param data
	 *            the data of the event to build
	 * @return the event as an <code>int</code>
	 */
	public static int buildEvent(int type, int data) {
		return (type << TYPE_SHIFT) | data;
	}

	/**
	 * Returns the type of an event.
	 *
	 * @param event
	 *            an event
	 * @return event's type as an <code>int</code>
	 */
	public static int getType(int event) {
		return event >>> TYPE_SHIFT;
	}

	/**
	 * Returns the event's data.
	 *
	 * @param event
	 *            an event
	 * @return event's data as an <code>int</code>
	 */
	public static int getData(int event) {
		return event & ~(TYPE_MASK << TYPE_SHIFT);
	}

	public static int getTypeMask(int type) {
		return type << TYPE_SHIFT;
	}

	public static int getDataMask() {
		return ~(TYPE_MASK << TYPE_SHIFT);
	}
}

/**
 * This event generator is added to the system pool to replace an invalid user custom event generator. We are using this
 * generator to be sure to respect the order of the event generators in the system pool. If we don't add an invalid
 * generator when an user custom event generator is invalid, the next event generator will have the id of the missing
 * generator whereas the native side will continue to use the id set by the system microui generator. This generator
 * should stay in the system pool to not free its table index. Otherwise another generator can take its table index and
 * the native side will try to send unknown event for this new event generator!
 */
class InvalidEventGenerator extends GenericEventGenerator {

	@Override
	public int getEventType() {
		return -1; // no meaning
	}

	@Override
	public void setProperty(String name, String value) {
		// NOP
	}

	@Override
	protected void eventReceived(int event) {
		// NOP
	}

	@Override
	protected void eventsReceived(int[] events) {
		// NOP
	}
}
