/*
 * Java
 *
 * Copyright 2015-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import java.lang.Thread.UncaughtExceptionHandler;

import com.is2t.microbsp.microui.natives.NSystemPump;
import com.is2t.pump.Pump;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.microui.event.Event;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventHandler;
import ej.microui.event.generator.Buttons;
import ej.microui.event.generator.Command;
import ej.microui.event.generator.Pointer;
import ej.microui.event.generator.States;
import ej.trace.Tracer;

/**
 * Internal pump to retrieve native input events. The 8 first events identifiers are fixed in this class. Until 16
 * events (see sub classes) are allowed (MicroUI {@link Event} specification).
 * <p>
 * This is necessary to ensure to the application a full consistency of the events sequences. With the combination of
 * several policies, it tries to avoid to have a full queue. Otherwise, there are too much events sent than events
 * executed. There is a global problem of application calibration, and an {@link MicroUIException} is thrown is thrown
 * if an application is not well calibrated.
 */
public class MicroUIPump implements Pump, EventHandler {

	/**
	 * Event Generator {@link Command#EVENT_TYPE}. Fixed by MicroUI API.
	 */
	public static final int COMMAND = Command.EVENT_TYPE;

	/**
	 * Event Generator {@link Buttons#EVENT_TYPE}. Fixed by MicroUI API.
	 */
	public static final int BUTTON = Buttons.EVENT_TYPE;

	/**
	 * Event Generator {@link Pointer#EVENT_TYPE}. Fixed by MicroUI API.
	 */
	public static final int POINTER = Pointer.EVENT_TYPE;

	/**
	 * Event Generator {@link States#EVENT_TYPE}. Fixed by MicroUI API.
	 */
	public static final int STATE = States.EVENT_TYPE;

	/**
	 * Unused event
	 */
	protected static final int EVENT_4 = 0x04;

	/**
	 * Internal event to manage {@link MicroUI#callSerially(Runnable)} event.
	 */
	public /* used by Display */ static final int CALLSERIALLY = 0x05;

	/**
	 * Internal event to stop the pump
	 */
	protected static final int STOP_PUMP = 0x06;

	/**
	 * Internal event to add native input event
	 */
	protected static final int EVENT_NATIVE = 0x07;

	private static final int NO_CURRENT_LOG = -1;
	private static final int INITIAL_EVENTS_DATA_ARRAY_SIZE = 1;

	/**
	 * Current log sent to logger. Allows to send an "end of event" log.
	 */
	protected int log;

	/**
	 * Data associated to current log.
	 */
	protected int logData;

	private final Thread thread;
	private volatile boolean isRunning;

	/**
	 * Stores objects that are linked with some software events like {@link Runnable}. The type of the object depends on
	 * the event. The very first implementation was an array that followed the super integer queue. But practically, a
	 * few number of events have an associated event data so in order to gain memory, it has been replaced by a simple
	 * growing array. Read/Write into this array must be synchronized on this instance.
	 *
	 * This array is static because it's known by native side
	 */
	protected static Object[] eventsData;
	private int nextEventsDataIndex; // next available index (0 by default)

	/**
	 * Creates the internal pump with a maximum of elements
	 */
	public MicroUIPump() {

		// Consider the pump is in the running state has soon as it is created.
		// The pump may be stopped BEFORE the thread running it is started.
		this.isRunning = true;

		// create the thread
		this.thread = new Thread(this, "UIPump"); //$NON-NLS-1$

		this.log = NO_CURRENT_LOG;
		eventsData = new Object[INITIAL_EVENTS_DATA_ARRAY_SIZE];

		NSystemPump.initialize();
	}

	/**
	 * Gets the MicroUI pump thread.
	 *
	 * @return the pump thread.
	 */
	public Thread getThread() {
		return this.thread;
	}

	/**
	 * Sets the priority.
	 *
	 * @param priority
	 *            the priority to set.
	 */
	public void setPriority(int priority) {
		this.thread.setPriority(priority);
	}

	/**
	 * Tells if pump is alive or not.
	 *
	 * @return true when the pump is alive.
	 */
	public boolean isDone() {
		return !this.thread.isAlive();
	}

	/**
	 * Starts the pump.
	 */
	public void start() {
		this.thread.setPriority(Constants.getInt(MicroUIProperties.CONSTANT_PUMP_PRIORITY));
		this.thread.start();
	}

	@Override
	public void stop() {
		this.isRunning = false;
		NSystemPump.interrupt();
	}

	@Override
	public void run() {

		// notify native worlt that the pump has started
		NSystemPump.start();

		while (this.isRunning) {
			try {
				// read data…
				int data = read();
				// … then execute it
				execute(data, SystemMicroUI.getType(data));
			} catch (Throwable e) {
				// something went wrong
				try {
					handleError(e);
				} catch (Throwable e2) {
					// handleError() crashes for an unknown reason
					// continue
				}
			} finally {
				executeDone();
			}
		}
	}

	/**
	 * Executes the pump event. Called in pump context (pump thread). Allow to read next event from queue
	 * ({@link #read()}.
	 *
	 * @param event
	 *            the event to execute
	 * @param eventType
	 *            the type of event
	 * @throws InterruptedException
	 *             thrown when pump is stopped
	 */
	private void execute(int event, int eventType) throws InterruptedException {

		switch (eventType) {

		case COMMAND:
		case BUTTON:
		case POINTER:
		case STATE:
			// Event Generator events
			executeEventGeneratorEvent(event);
			break;

		case EVENT_4:
			// Invalid events: drop them
			dropEvent(event);
			break;

		case MicroUIPump.CALLSERIALLY:
			((Runnable) (getEventData(event))).run();
			break;

		case STOP_PUMP:
			// Stop event:
			executeStopEvent();
			break;

		case EVENT_NATIVE:
			// Native input event
			executeInputEvent(event);
			break;

		default:
			// user event
			executeEvent(event, eventType);
			break;
		}
	}

	/**
	 * Lets sub-classes manage the event. By default this event to the pump's event handler.
	 *
	 * @param event
	 *            the event to execute
	 * @param eventType
	 *            the type of event
	 * @throws InterruptedException
	 *             thrown when pump is stopped
	 */
	protected void executeEvent(int event, int eventType) throws InterruptedException {
		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			this.log = Log.EXECUTE_USER_EVENT;
			this.logData = event;
			Log.Instance.recordEvent(this.log, this.logData);
		}
		callHandler(event, getPumpHandler());
	}

	/**
	 * Gets the pump's event handler: the pump is an event handler which can redirect serialized events to its handler.
	 * <p>
	 * By default there is no handler.
	 *
	 * @return the pump's event handler
	 */
	protected EventHandler getPumpHandler() {
		return null;
	}

	/**
	 * Called in pump context (pump thread) as soon as the event execution is done.
	 */
	protected void executeDone() {
		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			if (this.log != NO_CURRENT_LOG) {
				Log.Instance.recordEventEnd(this.log, this.logData);
				this.log = NO_CURRENT_LOG;
			}
		}
	}

	/**
	 * Called in pump context (pump thread) when the event execution throws an exception.
	 *
	 * @param e
	 *            the exception thrown by event handler during event execution.
	 */
	protected void handleError(Throwable e) {
		// catch any user errors
		MicroUI.MicroUI.errorLog(e);
	}

	/**
	 * Reads next event from queue. This method is called by pump itself and can be called again in pump context (pump
	 * thread) to read next event (when native event requires more than one event).
	 *
	 * @return the next queue event or lock the pump execution until a new event is available.
	 * @throws InterruptedException
	 *             thrown when pump is stopped
	 */
	public int read() throws InterruptedException {
		assert Thread.currentThread() == this.thread;
		return NSystemPump.read();
	}

	/**
	 * Returns the pump's event handler when the given event handler is this pump itself. In this case, the event is
	 * dispatched immediately to the pump's event handler (no have to stack again the event).
	 *
	 * When the given event handler is another instance (null allowed), it is returned.
	 *
	 * @param eventHandler
	 *            the event handler to check
	 * @return an event handler
	 */
	private @Nullable EventHandler fixEventHandler(@Nullable EventHandler eventHandler) {
		return eventHandler == this ? getPumpHandler() : eventHandler;
	}

	/**
	 * Executes the stop event.
	 *
	 * @see #STOP_PUMP
	 */
	protected void executeStopEvent() {
		// Stop event: pump has been stopped (do not log this event)
		// nothing to do actually
		assert !this.isRunning;
	}

	/**
	 * The event generator event has been serialized. Now, gives this event to the pump's event handler (and not to the
	 * event generator's handler because it is not the expected receiver).
	 *
	 * @param event
	 *            the event to dispatch
	 */
	private void executeEventGeneratorEvent(int event) {

		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			this.log = Log.EXECUTE_EVENTGENERATOR_EVENT;
			this.logData = Event.getType(event);
			Log.Instance.recordEvent(this.log, this.logData, Event.getGeneratorId(event), Event.getData(event));
		}

		callHandler(event, getPumpHandler());
	}

	/**
	 * Executes the input event. Can read additional input events, then convert event in a MicroUI event generator
	 * event. Finally, gives this event to the event generator's events handler. If this events handler is the pump
	 * itself or if is null, gives this event to the given default events handler (the pump's event handler).
	 *
	 * @param event
	 *            the input event to manage.
	 * @throws InterruptedException
	 *             thrown when pump is stopped
	 *
	 * @see #EVENT_NATIVE
	 */
	protected void executeInputEvent(int event) throws InterruptedException {

		EventGenerator converter;

		try {

			// retrieve event generator from event
			converter = Event.getGenerator(event);

		} catch (IllegalArgumentException e) {
			// [not nominal use-case] there is no converter for this event (MicroUI spec): drop this input event
			dropEvent(event);
			return;
		}

		// retrieve the event generator's event handler (throw a
		// NullPointerException if event generator is null)
		EventHandler handler = fixEventHandler(converter.getEventHandler());

		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			this.log = Log.EXECUTE_INPUT_EVENT;
			this.logData = converter.getEventType();
			Log.Instance.recordEvent(this.log, this.logData, converter.getId(), event);
		}

		// have to convert the event even if the handler is null because converter may
		// unstack another event
		converter.convert(this, event, handler);
	}

	/**
	 * Drops the event.
	 *
	 * @param event
	 *            the event to drop.
	 */
	private void dropEvent(int event) {
		if (Constants.getBoolean(Tracer.TRACE_ENABLED_CONSTANT_PROPERTY)) {
			Log.Instance.recordEvent(Log.EXECUTE_DROP_EVENT, event);
		}
	}

	/**
	 * Dispatchs event to the given event handler.
	 *
	 * @param event
	 *            the event to dispatch.
	 * @param handler
	 *            the event handler to call.
	 */
	public static void callHandler(int event, @Nullable EventHandler handler) {
		try {
			handler.handleEvent(event);
		} catch (NullPointerException e) {
			// NPE possible causes: handler is null or a NPE occurs in handleEvent
			if (handler != null) {
				// NPE occurs in handleEvent: throw this error
				throw e;
			}
			// else: not standard use-case: most of time, the event handler is not null
			// -> nothing to do
		}
	}

	/**
	 * Retrieves and consumes the event data associated with the given event. Assumes the event's index has been
	 * returned by {@link #storeEventData(Object)}.
	 *
	 * @param event
	 *            the event which contains the link to the data to return.
	 * @return the event data associated with the given event
	 */
	protected synchronized Object getEventData(int event) {
		int index = SystemMicroUI.getData(event);
		Object eventData = eventsData[index];
		eventsData[index] = null;
		return eventData;
	}

	/**
	 * Read (not consumed) the event data associated with the given event. Assume the event's index has been returned by
	 * {@link #storeEventData(Object)}.
	 *
	 * @param event
	 *            the event which contains the link to the data to return.
	 * @return the event data associated with the given event
	 */
	protected synchronized Object readEventData(int event) {
		int index = SystemMicroUI.getData(event);
		return eventsData[index];
	}

	/**
	 * Stores event data and returns the index int events array. If the array is full, it is resized. The maximum number
	 * of event data (65536) CANNOT be reached because at any time, the number of indexes used cannot be greater that
	 * the size of the queue.
	 * <p>
	 * Caller MUST be the owner of the pump monitor.
	 *
	 * @param data
	 *            the object to store in events array
	 * @return the object index in events array
	 */
	protected int storeEventData(Object data) {
		if (eventsData[this.nextEventsDataIndex] != null) {
			Object[] oldEventsData = eventsData;
			int oldLength = oldEventsData.length;
			// find a null entry (maybe the array has grown recently and is
			// fragmented with null entries)
			grow: {
				for (int i = -1; ++i < oldLength;) {
					if (oldEventsData[i] == null) {
						this.nextEventsDataIndex = i;
						break grow; // no need to grow
					}
				}

				// here, need to grow the array
				// elements MUST keep the same indexes within the new array
				// because their index is stored in a DISPLAY event
				eventsData = new Object[oldLength + 1];
				System.arraycopy(oldEventsData, 0, eventsData, 0, oldLength);
				this.nextEventsDataIndex = oldLength;
			}
		}
		int index = this.nextEventsDataIndex;
		eventsData[index] = data;
		this.nextEventsDataIndex = (index + 1) % eventsData.length;
		return index;
	}

	/**
	 * Removes all events & event data in the buffer.
	 */
	public synchronized void clearQueue() {
		NSystemPump.clearQueue();
		for (int i = eventsData.length; --i >= 0;) {
			eventsData[i] = null;
		}
		this.nextEventsDataIndex = 0;
	}

	/**
	 * Called by the application, by {@link #createAndHandleEvent(int)} or {@link #createAndHandleEvent(int, Object)}.
	 * Adds an event in the queue.
	 *
	 * Serializes this event even if caller is the pump itself: during a render or an event execution, the application
	 * wants to serialize a new event.
	 *
	 * @throws MicroUIException
	 *             if the event could not be added because the queue is full
	 */
	@Override
	public boolean handleEvent(int event) {
		if (!NSystemPump.add(event)) {
			throw new MicroUIException(MicroUIException.OUT_OF_EVENTS);
		}
		return true;
	}

	/**
	 * Creates an event without associated event data and adds it in the pump.
	 *
	 * @param event
	 *            the event to handle.
	 */
	public void createAndHandleEvent(int event) {
		handleEvent(SystemMicroUI.buildEvent(event, 0));
	}

	/**
	 * Creates an event with associated event data and adds it in the pump.
	 *
	 * @param event
	 *            the event to handle.
	 * @param eventData
	 *            the event data.
	 */
	public synchronized void createAndHandleEvent(int event, Object eventData) {
		try {
			event = createEvent(event, eventData);
			handleEvent(event);
		} catch (MicroUIException e) {
			// event was not added to the queue - remove the associated eventData
			eventsData[SystemMicroUI.getData(event)] = null;
			throw e;
		}
	}

	/**
	 * Creates an event with associated event data.
	 * <p>
	 * Caller MUST be the owner of the pump monitor.
	 *
	 * @param event
	 *            the event to handle.
	 * @param eventData
	 *            the event data.
	 * @return the created event
	 */
	protected int createEvent(int event, Object eventData) {
		int eventDataIndex = storeEventData(eventData);
		return SystemMicroUI.buildEvent(event, eventDataIndex);
	}

	/**
	 * @return pump | default exception handler
	 */
	@Nullable
	public static UncaughtExceptionHandler getUncaughtExceptionHandler() {
		UncaughtExceptionHandler handler = Thread.currentThread().getUncaughtExceptionHandler();
		return null == handler ? Thread.getDefaultUncaughtExceptionHandler() : handler;
	}
}