/*
 * Java
 *
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import ej.annotation.Nullable;
import ej.microui.display.Display;
import ej.microui.display.DisplayPump;
import ej.microui.event.EventGenerator;
import ej.microui.event.EventHandler;

public class MicroUIFactory {

	protected static MicroUIFactory INSTANCE;
	private @Nullable MicroUIPump pump;

	// known by system microui generator
	public static MicroUIFactory getInstance() {
		MicroUIFactory instance = INSTANCE;
		if (instance == null) {
			// no injection - default case
			INSTANCE = instance = new MicroUIFactory();
		}
		return instance;
	}

	/**
	 * Pump created for the display
	 *
	 * @param d
	 *            the display when the pump will run for
	 * @return the created pump
	 */
	public final DisplayPump createPump(Display d) {
		DisplayPump pump = newDisplayPump(d);
		setPump(pump);
		return pump;
	}

	/**
	 * See KF implementation
	 */
	protected DisplayPump newDisplayPump(Display d) {
		return new DisplayPump();
	}

	/**
	 * Pump created for the input when there is no display on the platform. Called
	 * by the System MicroUI Generator.
	 *
	 * @return the created pump
	 */
	public MicroUIPump createPump() {
		MicroUIPump pump = new MicroUIPump();
		setPump(pump);
		return pump;
	}

	private void setPump(MicroUIPump pump) {
		assert this.pump == null;
		this.pump = pump;
	}

	public MicroUIPump getPump() {
		return this.pump;
	}

	public void setEventHandler(EventGenerator gen, @Nullable EventHandler handler) {
		gen.checkEventHandlerPermission();
		gen.setDefaultEventHandler0(handler);
		gen.setEventHandler0(handler);
	}

	public void resetEventHandlers(Display display) {
		// does nothing by default
	}

	public void start(MicroUI microUI) {
		// initialize system
		microUI.init();

		try {
			this.pump.start();
		} catch (NullPointerException e) {
			// [not standard use-case] there is no display and no event generator on the
			// platform
		}
	}

	public void finalizeStartup(MicroUI microUI) {
		// start the system
		microUI.systemStart();
	}

	public void stop(final MicroUI microUI) {

		// stop the system
		microUI.systemStop();

		try {
			// first part : synchronous
			// stop pump
			this.pump.stop();

			// second part : asynchronous
			// Must be sure that the current thread is not one of the thread that
			// we will be waiting for its stop.
			new Thread() {
				@Override
				public void run() {
					microUI.waitForPump(MicroUIFactory.this.pump);
				}
			}.start();
		} catch (NullPointerException e) {
			// [not standard use-case] there is no display and no event generator on the
			// platform
		}

		// destroy system
		microUI.destroy();
	}
}
