/*
 * Java
 *
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import ej.error.Message;

/**
 * microUI-API
 */
public class MicroUIException extends RuntimeException {

	private static final long serialVersionUID = 1L;

	/**
	 * microUI-API
	 */
	public static final int MICROUI_NOT_STARTED = -1;

	/**
	 * microUI-API
	 */
	public static final int GENERIC_EVENT_GENERATOR_INVALID_CLASSNAME = -2;

	/**
	 * microUI-API
	 */
	public static final int DISPLAY_DEADLOCK = -3;

	/**
	 * microUI-API
	 */
	public static final int RESOURCE_INVALID_PATH = -4;

	/**
	 * microUI-API
	 */
	public static final int RESOURCE_INVALID_FILE = -5;

	/**
	 * microUI-API
	 */
	public static final int RESOURCE_CLOSED = -6;

	/**
	 * microUI-API
	 */
	public static final int IMAGE_OUT_OF_MEMORY = -7;

	/**
	 * microUI-API
	 */
	public static final int IMAGE_UNKNOWN_FORMAT = -8;

	/**
	 * microUI-API
	 */
	public static final int OUT_OF_EVENTS = -9;

	/**
	 * microUI-API
	 */
	public static final int NO_DISPLAY = -10;

	/**
	 * microUI-API
	 */
	public static final int NO_FONT = -11;

	/**
	 * microUI-API
	 */
	public static final int EVENTGENERATOR_POOL_FULL = -12;

	/**
	 * microUI-API
	 */
	public static final int DRAWING_ERROR = -13;

	/**
	 * microUI-API
	 */
	public static final int IMAGE_UNSUPPORTED_FORMAT = -14;

	/**
	 * XXX
	 */
	public static final int INTERN_NO_MICROUI_IMPL = 0;

	private final int errorCode;

	/**
	 * microUI-API
	 *
	 * @param errorCode
	 *            microUI-API
	 */
	public MicroUIException(int errorCode) {
		super(getMessage(errorCode));
		this.errorCode = errorCode;
	}

	/**
	 * Constructs an ImageCreationException with the specified detail message and optional value.
	 *
	 * @param errorCode
	 *            the exception identifier
	 * @param value
	 *            the optional exception message value
	 */
	public MicroUIException(int errorCode, String value) {
		super(getMessage(errorCode, value));
		this.errorCode = errorCode;
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public int getErrorCode() {
		return this.errorCode;
	}

	private static String getMessage(int errorCode) {
		return Message.at(new MicroUIErrorMessages(), errorCode);
	}

	private static String getMessage(int errorCode, String value) {
		return Message.at(new MicroUIErrorMessages(), errorCode, value);
	}

	/**
	 * Trims and checks the given resource path format and throw a {@link MicroUIException} if the format is not valid.
	 *
	 * @param path
	 *            the path to check
	 * @return the trimmed path
	 */
	public static String checkResourcePath(String path) {
		path = path.trim();
		if (path.length() == 0 || path.charAt(0) != '/') {
			// spec: only absolute identifier
			throw new MicroUIException(MicroUIException.RESOURCE_INVALID_PATH, path);
		}
		return path;
	}
}
