/*
 * Java
 *
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui;

import java.lang.Thread.UncaughtExceptionHandler;

import ej.bon.Constants;
import ej.bon.Util;

/**
 * microUI-API
 */
public abstract class MicroUI {

	private static final int ETERNITY = 5000;// 5 seconds (should be enough !!!)

	public static MicroUI MicroUI; // never null, by default it is a MicroUI instance

	static {
		// try to create System MicroUI Implementation
		MicroUI = getMicroUIImpl();
	}

	// field is synchronized using MicroUI instance
	private boolean isRunning;

	/**
	 * microUI-API
	 */
	@SuppressWarnings("unused")
	private MicroUI() {
		this(false);
	}

	/* default */ MicroUI(boolean dummy) {
		Log.create();
	}

	/**
	 * microUI-API
	 */
	public static void start() {
		// get MicroUI instance
		MicroUI microUI = MicroUI;
		synchronized (microUI) {
			if (!microUI.isRunning) {
				if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
					SecurityManager sm = System.getSecurityManager();
					if (sm != null) {
						sm.checkPermission(new MicroUIPermission(MicroUIPermission.ACTION_START));
					}
				}

				MicroUIFactory.getInstance().start(microUI);
				microUI.isRunning = true; // must start only once
				MicroUIFactory.getInstance().finalizeStartup(microUI);
			}
		}
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public static boolean isStarted() {
		return MicroUI.isRunning;
	}

	/**
	 * microUI-API
	 */
	public static void stop() {
		// get MicroUI instance
		MicroUI microUI = MicroUI;
		synchronized (microUI) {
			if (microUI.isRunning) {
				if (Constants.getBoolean(MicroUIProperties.CONSTANT_USE_SECURITYMANAGER)) {
					SecurityManager sm = System.getSecurityManager();
					if (sm != null) {
						sm.checkPermission(new MicroUIPermission(MicroUIPermission.ACTION_STOP));
					}
				}
				MicroUIFactory.getInstance().stop(microUI);
			}
		}
	}

	/* default */ abstract void init();

	/* default */ abstract void destroy();

	/* default */ abstract void systemStart();

	/* default */ abstract void systemStop();

	/* default */ abstract void systemClose();

	/**
	 * Gives the exception to the registered {@link UncaughtExceptionHandler} (if any) or just prints the exception.
	 *
	 * @param e
	 *            the exception
	 */
	public void errorLog(Throwable e) {
		UncaughtExceptionHandler handler = MicroUIPump.getUncaughtExceptionHandler();
		if (null != handler) {
			handler.uncaughtException(Thread.currentThread(), e);
		} else {
			e.printStackTrace();
		}
	}

	/**
	 * Prints the exception's message.
	 *
	 * @param e
	 *            the exception
	 */
	public void warning(Throwable e) {
		System.err.println(e.getMessage());
	}

	public static void checkRunning() {
		if (!MicroUI.isRunning) {
			throw new MicroUIException(MicroUIException.MICROUI_NOT_STARTED);
		}
	}

	/**
	 * Get the user's microui system implementation.
	 *
	 * @return a bsp implementation
	 */
	private static MicroUI getMicroUIImpl() {
		try {
			return (MicroUI) Constants.getClass(MicroUIProperties.CONSTANT_CLASS_MICROUI_IMPL).newInstance();
		} catch (Throwable ex) {
			// This error can occur if:
			// - muiSysClassname is invalid
			// - muiSysClassname is not a class
			// - muiSysClassname is not a MicroUI instance
			// - S3: HIL mock error (not connected)
			MicroUIException exception = new MicroUIException(MicroUIException.INTERN_NO_MICROUI_IMPL);
			exception.initCause(ex);
			exception.printStackTrace();
			return null;
		}
	}

	protected void waitForPump(MicroUIPump pump) {
		long start = Util.platformTimeMillis();

		boolean isDone = false;
		do {
			// set all done to true
			Thread.yield();
			isDone = pump.isDone();

			// watchdog...
			long delta = Util.platformTimeMillis() - start;
			if (delta >= ETERNITY) {
				isDone = true; // DONE. cannot wait more…
			}

		} while (!isDone);

		synchronized (MicroUI) {
			// close system resources
			systemClose();
			this.isRunning = false;
		}
	}

	/**
	 * microUI-API
	 *
	 * @param thread
	 *            microUI-API
	 * @return microUI-API
	 */
	public static boolean isUIThread(Thread thread) {
		assert thread != null;
		try {
			return thread == MicroUIFactory.getInstance().getPump().getThread();
		} catch (NullPointerException e) {
			checkRunning();
			throw e;
		}
	}

	/**
	 * microUI-API
	 *
	 * @return microUI-API
	 */
	public static boolean isUIThread() {
		try {
			return isUIThread(Thread.currentThread());
		} catch (NullPointerException e) {
			checkRunning();
			throw e;
		}
	}

	/**
	 * microUI-API
	 *
	 * @param run
	 *            microUI-API
	 */
	public static void callSerially(Runnable run) {
		assert run != null;
		try {
			MicroUIFactory.getInstance().getPump().createAndHandleEvent(MicroUIPump.CALLSERIALLY, run);
		} catch (NullPointerException e) {
			checkRunning();
			throw e;
		}
	}
}
