/*
 * Java
 *
 * Copyright 2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.microui.watchdog;

import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.lang.Watchdog;

/**
 * KF watchdog is capable of killing a feature as corrective action:
 * <ul>
 * <li>if Kernel triggered watchdog, does nothing,</li>
 * <li>otherwise, if a feature triggered watchdog, it is killed.</li>
 * </ul>
 */
public class KFWatchdog extends Watchdog<Module> {

	private static final int STOP_FEATURE_TIMEOUT = 5000;

	/**
	 * Instantiates a new inactive watchdog.
	 */
	public KFWatchdog() {
		super();
	}

	@Override
	public void trigger() {
		Module module = this.getContext();
		if (module == null) {
			return;
		}

		if (module == Kernel.getInstance()) {
			// Should never occurs if kernel is safe
			return;
		}

		// Module is a feature for sure
		try {
			Feature feature = (Feature) module;
			stopFeature(feature, STOP_FEATURE_TIMEOUT);
		} catch (Throwable t) {
			// TODO handle error
			t.printStackTrace();
		}
	}

	/**
	 * KF-1.3 spec: <code>Feature.stop()</code> may result to <code>STOPPED</code> state when there are remaining
	 * references from Kernel to the Feature being killed. Wadapps API does not provide this intermediate step (the
	 * <code>STOPPED</code> state is not a Wadapps state). This function blocks until all Feature references are removed
	 * by the Kernel (i.e. wait until pumps have finished their current job, ...) or the timeout occurs.
	 *
	 * @param f
	 *            the feature to stop.
	 * @param timeout
	 *            the maximum time to wait in milliseconds.
	 * @return true if Feature has been successfully stopped, false if timeout occurred (the Feature remains in
	 *         <code>STOPPED</code> state)
	 */
	public static boolean stopFeature(Feature f, long timeout) {
		int delay = 250;
		long start = System.currentTimeMillis();
		while (true) {
			try {
				f.stop();
			} catch (IllegalStateException e) {
				// an other thread may have stopped or uninstalled the Feature.
			}
			if (f.getState() != Feature.State.STOPPED) {
				// may be in INSTALLED state
				// may also be in STARTED or UNINSTALLED if done by an other thread
				// => terminate
				return true;
			}
			if (System.currentTimeMillis() - start >= timeout) {
				return false;
			}

			try {
				Thread.sleep(delay);
			} catch (InterruptedException e) {
				// continue
			}
		}
	}

}
