/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.is2t.microbsp.microui.natives;

import ej.annotation.Nullable;
import ej.microui.MicroUIException;

/**
 * Methods to prepare native method parameters or convert native method returned values.
 *
 * Prevent to write some non-native methods in NSystemxxx classes (useful to implement MicroUI over J2SE).
 */
public class NTools {

	public static String getRawImagePath(String path) {
		return path + NSystemDisplay.IMAGE_RAW_SUFFIX;
	}

	public static String getExternalImagePath(String path) {
		return path + "\0"; //$NON-NLS-1$
	}

	/**
	 * Throws an exception according the native return code.
	 *
	 * @param ret
	 *            the native return code
	 * @param path
	 *            the path of the image to load or null
	 * @return the native return code
	 */
	public static final int checkError(int ret, @Nullable String path) {

		int error;
		switch (ret) {
		case NSystemDisplay.OUT_OF_MEMORY:
			// not enough room in images heap
			error = MicroUIException.IMAGE_OUT_OF_MEMORY;
			break;
		case NSystemDisplay.INVALID_TYPE:
			// the native side cannot decode this kind of image
			error = MicroUIException.IMAGE_UNKNOWN_FORMAT;
			break;
		case NSystemDisplay.UNSUPPORTED_FORMAT:
			// the native side cannot create this kind of image
			error = MicroUIException.IMAGE_UNSUPPORTED_FORMAT;
			break;
		case NSystemDisplay.INVALID_PATH:
			// cannot retrieve the image
			error = MicroUIException.RESOURCE_INVALID_PATH;
			break;
		case NSystemDisplay.INVALID_FILE:
			// there was an error to open the png image
			error = MicroUIException.RESOURCE_INVALID_FILE;
			break;
		default:
			if (ret >= 0) {
				// no error
				return ret;
			}
			// unknown error
			error = ret;
			break;
		}

		if (null == path) {
			throw new MicroUIException(error);
		} else {
			throw new MicroUIException(error, path);
		}
	}
}
