/*
 * Java
 *
 * Copyright 2010-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.is2t.microbsp.microui.natives;

import com.is2t.vm.support.io.MemoryInputStream;

import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.SImageMetadata;

public class NSystemDisplay {

	/*
	 * Native error constants
	 *
	 * WARNING copied in microui-extended
	 */

	/**
	 * no error
	 */
	public static final int NO_ERROR = 0;

	/**
	 * Native side cannot decode the file
	 */
	public static final int INVALID_TYPE = -1;

	/**
	 * oom
	 */
	public static final int OUT_OF_MEMORY = -2;

	/**
	 * File content is invalid
	 */
	public static final int INVALID_FILE = -3;

	/**
	 * File not found
	 */
	public static final int INVALID_PATH = -4;

	/**
	 * File has been closed
	 */
	public static final int FILE_CLOSED = -6;

	/**
	 * Native side cannot create the graphics context
	 */
	public static final int UNSUPPORTED_FORMAT = -7;

	/**
	 * Anonymous error
	 */
	public static final int NOK = -9;

	/**
	 * Used to open image in default RAW format (default image format)
	 */
	public static final int DEFAULT_FORMAT = -1;

	/**
	 * invalid event to identify a receiver (event type 0 is "COMMAND")
	 */
	public static final int NO_END_OF_FLUSH_EVENT = 0;

	/**
	 * Suffix to apply on image path to load a RAW image.
	 */
	public static final String IMAGE_RAW_SUFFIX = "_raw"; //$NON-NLS-1$

	private NSystemDisplay() {
		// not entry point
	}

	public static native int initAndGetNextGraphicsContextId(byte[] sd, int sdSize, char[] internStringsArray);

	/**
	 * Configures the log group for the event engine
	 *
	 * @param groupID
	 *            MicroUI group identifier
	 */
	public static native void setTraceGroupID(int groupID);

	public static native int getNbBitsPerPixel();

	public static native boolean isColor();

	public static native int getNumberOfColors();

	public static native boolean isDoubleBuffered();

	public static native boolean hasBacklight();

	/**
	 * LCD manipulations
	 */
	public static native void setContrast(int contrast);

	public static native int getContrast();

	public static native void setBacklight(int backlight);

	public static native int getBacklight();

	/**
	 * LCD offscreens
	 *
	 * @param width
	 *            positive: standard offscreen, negative: virtual image
	 * @param height
	 *            positive: standard offscreen, negative: virtual image
	 * @param dstSD
	 *            the sni context of destination
	 * @param format
	 *            the format of the offscreen
	 * @return an error code, 0 on success
	 */
	public static native int openOffScreen(int width, int height, byte[] dstSD, byte format);

	public static native void closeOffScreen(byte[] sd);

	public static native void flushScreen(byte[] sd);

	public static native void waitFlush();

	/**
	 * LCD rendering
	 */

	public static native boolean isTransparent(byte[] sd);

	/**
	 * LCD read
	 */
	public static native int readPixel(byte[] sd, int x, int y);

	public static native void getARGB(byte[] srcSd, int[] rgbData, int offset, int scanlength, int x, int y, int width,
			int height);

	/**
	 * Side-effects on drawings
	 */
	public static native int getDisplayColor(int color);// alpha must be 0

	/**
	 * This method MUST BE called only by {@link GraphicsContext#resetClip()}
	 * <p>
	 * Reset the current native clip to its maximum bounds
	 */
	public static native void resetClip(byte[] sd);

	/**
	 * This method MUST BE called only by {@link GraphicsContext#setClip(int, int, int, int)} !!!
	 * <p>
	 * Set a clip smaller or equal than current clip (update current native clip)
	 */
	public static native void setClip(byte[] sd, int x, int y, int width, int height);

	/**
	 * This method MUST BE called only by {@link GraphicsContext#intersectClip(int, int, int, int)} !!!
	 * <p>
	 * Set a clip smaller or equal than given wrapper clip (replace current native clip without checking its bounds
	 * (checked by caller))
	 */
	public static native void intersectClip(byte[] sd, int x, int y, int width, int height);

	/**
	 * @param color
	 *            fully opaque 32-bit color
	 */
	public static native void setForegroundColor(byte[] sd, int color);

	/**
	 * @param color
	 *            24-bit color
	 * @param active
	 *            false to disable the background color, true to apply the background color
	 *
	 */
	public static native void setBackgroundColor(byte[] sd, int color, boolean active);

	/**
	 * @param width
	 *            0 to disable the ellipsis, positive value to enable it
	 */
	public static native void setEllipsis(byte[] sd, int width);

	/**
	 * Gets and clears the drawing log flags from a graphics context.
	 *
	 * The flags are reset to their success value after retrieving their value.
	 *
	 * @param sd
	 *            the graphics context
	 * @return the drawing log flags
	 */
	public static native int getAndClearDrawingLogFlags(byte[] sd);

	/**
	 * @param dstSD
	 *            null to check image path only, an {@link SImageMetadata} byte array to load the image
	 * @return an error code (see Image error codes)
	 */
	public static native int loadImageFromPath(byte[] path, int length, boolean dynamic, boolean externalResource,
			boolean rawAndByteAddressableRequired, int format, @Nullable byte[] dstSD);

	public static native int loadImageFromBytes(byte[] data, int offset, int length, int format, byte[] dstSD);

	public static native int loadImageFromMIS(MemoryInputStream stream, int size, int format, byte[] dstSD);

	/**
	 * Fonts
	 */
	public static native int addInternalFont(byte[] path, int pathLength, byte[] fontData);

	public static native int getFontIdentifiers(byte[] fontData, int[] identifiers);

	/**
	 * Gives the next "end of flush" event to add in pump queue at the end of next fluh.
	 *
	 * @param event
	 *            the event to add into the pump queue
	 * @return the old "end of flush" event (if exists)
	 */
	public static native int addEndOfFlushEvent(int event);

	/**
	 * Links an existing native image (offscreen which points to a buffer located in images heap or in flash).
	 *
	 * @param sd
	 *            new image data to fill with native image
	 * @param imageReference
	 *            native image reference
	 * @param freed
	 *            true if original image (native image) has to be freed or not (this peace of information was only
	 *            available in object {@link Image} but this object has been garbaged collected).
	 *
	 * @deprecated Kept for backward compatibility (widget2 over UI3). Task M0092MEJAUI-2065.
	 */
	@Deprecated
	public static native void linkImage(byte[] sd, int imageReference, boolean freed);

	/**
	 * @see GraphicsContext#notifyDrawingRegion()
	 * @param sd
	 *            the graphics context to notify
	 */
	public static native void notifyDrawingRegion(byte[] sd);

}