/*
 * Java
 *
 * Copyright 2015-2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.pump._int;

import com.is2t.pump.Pump;

/**
 * Int based pump.
 */
public abstract class IntPump implements Pump {

	private volatile boolean isRunning;

	public IntPump(){
		// Consider the pump is in the running state has soon as it is created.
		// The pump may be stopped BEFORE the thread running it is started.
		isRunning = true;
	}

	@Override
	public void run() {
		while (this.isRunning) {
			try {
				// read data…
				int data = read();
				// … then execute it
				execute(data);
			} catch (InterruptedException e) {
			} catch (Throwable e) {
				// something went wrong
				try {
					crash(e);
				} catch (Throwable e2) {
					// crash() crashes for an unknown reason
					// continue
				}
			}
		}
	}

	@Override
	public void stop() {
		this.isRunning = false;
	}

	/**
	 * Reads a data.
	 *
	 * @return the read data.
	 */
	public abstract int read() throws InterruptedException;

	/**
	 * Executes last read data.
	 *
	 * @param data
	 *            the data to execute.
	 * @throws InterruptedException
	 */
	protected abstract void execute(int data) throws InterruptedException;

	/**
	 * An exception occurs while pumping.
	 * <p>
	 * Default behavior is to stop the pump and print the stack trace.
	 *
	 * @param throwable
	 *            the throwable thrown while pumping.
	 */
	protected void crash(Throwable throwable) {
		stop();
		throwable.printStackTrace();
	}

}
