/*
 * Java
 *
 * Copyright 2010-2019 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.pump._int;

/**
 * Characteristics:
 * <ul>
 * <li>Fixed buffer size</li>
 * <li>Simple synchronization on buffer array</li>
 * <li>A {@link QueueFullException} is thrown if the queue is full</li>
 * </ul>
 */
public class FIFOIntQueue implements BlockingIntQueue {

	/**
	 * Circular buffer.
	 */
	protected final int[] buffer;
	/**
	 * Next to read.
	 */
	protected int ptrBegin;
	/**
	 * Next to add.
	 */
	protected int ptrEnd;

	public FIFOIntQueue(int size) {
		this.buffer = new int[size + 1]; // +1: one index in the queue is always empty
	}

	@Override
	public int poll() throws InterruptedException {
		synchronized (this.buffer) {
			while (this.ptrBegin == this.ptrEnd) {
				this.buffer.wait();
			}
			int data = this.buffer[this.ptrBegin];
			this.buffer[this.ptrBegin] = 0;
			this.ptrBegin = getNext(ptrBegin);
			return data;
		}
	}

	@Override
	public boolean add(int value) {
		if (!offer(value)) {
			throw new IllegalStateException();
		}
		return true;
	}

	@Override
	public boolean offer(int value) {
		synchronized (this.buffer) {
			// check if the queue is not full
			int nextEnd = getNext(this.ptrEnd);
			if (nextEnd == this.ptrBegin) {
				// FIFO is full
				return false;
			}
			// add the new event at the last index: ptrEnd
			this.buffer[this.ptrEnd] = value;
			this.ptrEnd = nextEnd;
			// notify the waiting thread
			this.buffer.notify();
			return true;
		}
	}

	private int getNext(int ptr) {
		return ++ptr == this.buffer.length ? 0 : ptr;
	}
}
