/*
 * Java
 *
 * Copyright 2019-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import ej.kf.AlreadyLoadedFeatureException;
import ej.kf.Feature;
import ej.kf.IncompatibleFeatureException;
import ej.kf.InvalidFormatException;
import ej.kf.Kernel;

/**
 * Implementation of Feature loader on Simulator. This loader is able to load
 * Features:
 * <li>from the Feature classpath, when data is the name of the Feature to be
 * loaded</li>
 * <li>from an InputStream, when data is a FS3 (which is a JAR)</li>
 */
public class DynamicFeatureLoader implements IFeatureLoader {

	private static final int MAX_FEATURE_NAME = 256;

	private static final int ERROR_FORMAT_MAX_FEATURE_NAME = InvalidFormatException.FIRST_SPECIFIC_ERROR;
	private static final int ERROR_FORMAT_STATIC_FEATURE_NOT_FOUND = InvalidFormatException.FIRST_SPECIFIC_ERROR + 1;

	/** The JAR magic numbers. */
	private static final byte[] SIGNATURE = {0x50 , 0x4B, 0x03, 0x04};

	@Override
	public boolean canLoad(byte[] signature) {
		// either load by name or load FS3
		return true;
	}

	private boolean isFS3(byte[] signature) {
		if (signature.length < SIGNATURE.length) {
			return false;
		}

		for (int i = 0; i < SIGNATURE.length; i++) {
			if (signature[i] != SIGNATURE[i]) {
				return false;
			}
		}

		return true;
	}

	@Override
	public boolean canUnload() {
		return true;
	}

	@Override
	public Feature load(byte[] signature, InputStream is)
			throws IOException, InvalidFormatException, IncompatibleFeatureException, AlreadyLoadedFeatureException {
		if (isFS3(signature)) {
			return loadFS3(is);
		} else {
			return loadFromFeatureClasspath(signature, is);
		}
	}

	@Override
	public boolean unload(Feature feature) {
		if (unloadFS3(feature)) {
			return true;
		} else {
			return false;
		}
	}

	@Override
	public void loadInstalledFeatures() {
		// instantiate installed Features (post-link firmware)
		int nbInstalledFeatures = DynamicLoaderNatives.getMaxNbInstalledFeatures();
		for (int i = -1; ++i < nbInstalledFeatures;) {
			long featureSectionDescriptor = DynamicLoaderNatives.getInstalledFeatureDescriptor(i);
			if (featureSectionDescriptor != -1) {
				Feature f = new Feature(this, featureSectionDescriptor, Feature.FEATURE_HANDLE_NONE);
				Kernel.addInstalledFeature(f);
			}
		}
	}

	private Feature loadFS3(InputStream is) throws IOException {
		final int id = DynamicLoaderNatives.openDynamicFeatureTransfer();

		DynamicLoaderNatives.transferDynamicFeature(id, SIGNATURE, SIGNATURE.length);

		byte[] buffer = new byte[1024];
		int read = is.read(buffer);
		while (read != -1) {
			DynamicLoaderNatives.transferDynamicFeature(id, buffer, read);
			read = is.read(buffer);
		}

		long vmDescriptor = DynamicLoaderNatives.closeDynamicFeatureTransfer(id);
		return new Feature(this, vmDescriptor, Feature.FEATURE_HANDLE_NONE);
	}


	private Feature loadFromFeatureClasspath(byte[] signature, InputStream is)
			throws IOException, AlreadyLoadedFeatureException, InvalidFormatException {
		InputStreamReader r = new InputStreamReader(is);
		char[] chars = new char[MAX_FEATURE_NAME + 1];
		int signatureLength = signature.length;
		int totalNbRead = 0;
		for (int i = -1; ++i < signatureLength;) {
			char c = (char) (signature[i] & 0xff);
			if (c != 0) {
				chars[totalNbRead++] = c;
			} else {
				// name is lower than 4 bytes
				break;
			}
		}

		int followingNbRead = r.read(chars, totalNbRead, MAX_FEATURE_NAME - totalNbRead);
		if (followingNbRead != -1) {
			totalNbRead += followingNbRead;
		}

		if (totalNbRead > MAX_FEATURE_NAME) {
			throw new InvalidFormatException(ERROR_FORMAT_MAX_FEATURE_NAME);
		}

		long featureSectionDescriptor = DynamicLoaderNatives.loadStaticFeatureByName(chars, totalNbRead);

		if (featureSectionDescriptor >= 0) {
			// successfully loaded
			return new Feature(this, featureSectionDescriptor, Feature.FEATURE_HANDLE_NONE);
		} else {
			// error
			switch ((int) featureSectionDescriptor) {
			case DynamicLoaderNatives.ERROR_LOADSTATIC_ALREADYLOADED: {
				// Get back the Feature
				Feature f = Kernel.getFeatureByName(new String(chars, 0, totalNbRead));
				assert (f != null);
				throw new AlreadyLoadedFeatureException(f);
			}
			case DynamicLoaderNatives.ERROR_LOADSTATIC_NOT_FOUND:
				throw new InvalidFormatException(ERROR_FORMAT_STATIC_FEATURE_NOT_FOUND);
			default:
				throw new RuntimeException();
			}
		}
	}

	private boolean unloadFS3(Feature feature) {
		return DynamicLoaderNatives.unloadDynamicFeature(feature);
	}

}
