/*
 * Java
 *
 * Copyright 2022-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

import com.is2t.kf.elflw.AllocationSection;
import com.is2t.kf.elflw.DynamicAllocationSection;
import com.is2t.kf.elflw.ElfConstants;
import com.is2t.kf.elflw.Header;
import com.is2t.kf.elflw.ProgAllocationSection;
import com.is2t.kf.elflw.RelocationSectionSoar;
import com.is2t.kf.elflw.Section;
import com.is2t.kf.elflw.SectionHeader;
import com.is2t.kf.elflw.StringSection;
import com.is2t.kf.elflw.SymbolTableEntry;
import com.is2t.kf.elflw.SymbolTableSection;

import ej.annotation.Nullable;
import ej.bon.Constants;
import ej.kf.Feature;
import ej.kf.IncompatibleFeatureException;
import ej.kf.InvalidFormatException;

/**
 * ELF loader of <code>.fo</code> file.
 * <p>
 * Sections are organized in the following order:
 * <ul>
 * <li><code>.strtab</code>: symbol names</li>
 * <li><code>.symtab</code>: symbols</li>
 * <li><code>.bss.soar.feature</code>: RAM section</li>
 * <li><code>.kernel.uid: Kernel UID</li>
 * <li><code>.rodata.microej.resources</code>: ROM section (application resources with no relocations)</li>
 * <li><code>.rodata</code>: ROM section (application first code chunk 0)</li>
 * <li><code>.shstrtab</code>: section names</li>
 * <li><code>.debug.soar</code> (optional, only in <code>.fodbg</code> file)</li>
 * <li><code>.rel.debug.soar</code> (optional, only in <code>.fodbg</code> file)</li>
 * <li><code>.soar.rel</code>: relocations applied on firect code chunk 0</li>
 * <li><code>.rodata</code>: ROM section (application code chunk 1 with no relocations)</li>
 * <li><code>.rodata</code>: ROM section (application code chunk 2 with no relocations)</li>
 * <li>...</li>
 * <li><code>.rodata</code>: ROM section (application code chunk N)</li>
 * <li><code>.soar.rel</code>: relocations applied on chunk N</li>
 * <li><code>.rodata</code>: ROM section (application code chunk N+1)</li>
 * <li>...</li>
 * <ul>
 * This allows to process sections in streaming. (sections only depend to previously loaded sections).
 */
public class FoLoader implements ElfConstants, InvalidFormatConstants {

	// Constant names

	private static final String CONSTANT_PREFIX = "com.microej.runtime.kf.link.";
	/**
	 * Property suffix for architecture specific ELF symbol prefix.
	 * <p>
	 * See <code>init-kf-microjvm-dd/init.xml</code> script file.
	 */
	private static final String ARCH_SYMBOL_PREFIX_CONSTANT = CONSTANT_PREFIX + "symbol.prefix";
	private static final String DEBUG_CONSTANT = CONSTANT_PREFIX + "debug.enabled";
	private static final String RELOC_CHUNK_CONSTANT = CONSTANT_PREFIX + "chunk.relocations.count";
	private static final String TMP_TRANSFER_BUFFER_SIZE_CONSTANT = CONSTANT_PREFIX + "transferbuffer.size";

	// Symbol names

	private static final byte[] TYPES_HEADER_SYMBOL;
	private static final byte[] FEATURES_START_SYMBOL;

	static {
		String symbolPrefix = Constants.getString(ARCH_SYMBOL_PREFIX_CONSTANT);

		TYPES_HEADER_SYMBOL = (symbolPrefix + "_java_header_types_start").getBytes();
		FEATURES_START_SYMBOL = (symbolPrefix + "_java_features_start").getBytes();
	}

	// ELF Constants for <code>.fo</code> file

	/**
	 * The <code>.fo</code> file ELF machine type.
	 */
	private static final int EM_SOARMICROJVM = 0x2100;

	/**
	 * The <code>.fo</code> file ELF ABI version.
	 */
	private static final int SUPPORTED_ABI_VERSION = 0x03;

	/**
	 * The <code>.fo</code> file extended section type for SOAR relocation sections.
	 */
	private static final int SECTION_TYPE_REL_SOAR = ElfConstants.SHT_LOPROC;

	/**
	 * Number of logical sections in the <code>.fo</code> file.
	 * <li><code>.rodata</code> for application code. It may be physically divided in multiple ELF sections depending on
	 * the Feature code chunk size.</li>
	 * <li><code>.rodata.microej.resources</code> for application resources.</li>
	 */
	private static final int NB_ROM_SECTIONS_WITH_CHECKSUM = 2;

	private static final int STATICS_SECTION_INDEX = 3;
	private static final int KERNEL_UID_SECTION_INDEX = 4;
	private static final int RESOURCES_SECTION_INDEX = 5;
	private static final int FIRST_CODE_SECTION_INDEX = 6;

	/**
	 * The statically allocated RAM block for Feature code relocation.
	 */
	private final RAMBlock featureCodeChunkBlock;

	/**
	 * The offset of the next byte to read in the <code>.fo</code> input stream
	 */
	private int absoluteBytesPtr;

	/**
	 * FO ELF unit endianness
	 */
	private boolean lsbAndNotMsb;

	@Nullable
	private SectionHeader[] sectionHeaders;

	/**
	 * Direct link to the first encountered {@link SymbolTableSection}.
	 */
	@Nullable
	private SymbolTableSection symbolTableSection;

	/**
	 * This array follows the {@link SectionHeader} array. Contains the 0 based relative addresses of
	 * {@link ProgAllocationSection} from ROM and {@link DynamicAllocationSection} from RAM.
	 */
	@Nullable
	private int[] sectionsRelativeAdresses;

	/**
	 * This array follows the {@link SectionHeader} array. Contains a 32bits crc for {@link ProgAllocationSection}.
	 */
	@Nullable
	private int[] crcs;

	/**
	 * The current Feature handle (value not equals to {@link Feature#FEATURE_HANDLE_NONE}).
	 */
	private int featureHandle;

	/**
	 * Set when {@link #sectionHeadersLoaded(SectionHeader[])} is done.
	 */
	private int headerSize;

	/**
	 * Set when {@link #sectionHeadersLoaded(SectionHeader[])} is done.
	 */
	private long startAddressROM;

	/**
	 * Set when {@link #sectionHeadersLoaded(SectionHeader[])} is done.
	 */
	private long startAddressRAM;

	/**
	 * Set when {@link #sectionHeadersLoaded(SectionHeader[])} is done.
	 */
	private int sizeROM;

	/**
	 * Set when {@link #sectionHeadersLoaded(SectionHeader[])} is done.
	 */
	private int sizeRAM;

	/**
	 * The absolute address of {@link #TYPES_HEADER_SYMBOL}.
	 */
	private int typeHeaderAddress;

	/**
	 * The index of the current code section being relocated
	 */
	private int currentRelocatedCodeSectionIndex = SHN_UNDEF;

	public FoLoader() {
		this.featureHandle = Feature.FEATURE_HANDLE_NONE;
		long descriptor = DynamicLoaderNatives.getFeatureCodeChunk();
		assert (descriptor != 0);
		RAMBlock featureCodeChunkBlock = new RAMBlock(descriptor,
				DynamicLoaderNatives.getMemoryBlockStartAddress(descriptor),
				DynamicLoaderNatives.getMemoryBlockSize(descriptor));
		this.featureCodeChunkBlock = featureCodeChunkBlock;
	}

	/**
	 * Load a Relocatable unit.
	 *
	 * @param elfSignature
	 *            the ELF magic field already read (the first 4 bytes).
	 * @param is
	 *            the <code>.fo</code> input stream
	 * @throws IOException
	 *             on I/O errors
	 * @throws InvalidFormatException
	 *             if the Feature file cannot be loaded
	 * @throws IncompatibleFeatureException
	 *             if the Feature is not compatible with this Kernel
	 */
	public void loadFO(byte[] elfSignature, InputStream is)
			throws IOException, InvalidFormatException, IncompatibleFeatureException {
		try {
			load(elfSignature, is);
		} catch (IOException | IncompatibleFeatureException | InvalidFormatException e) {
			throw e;
		} catch (Throwable e) {
			if (Constants.getBoolean(DEBUG_CONSTANT)) {
				e.printStackTrace();
			}
			// any other error thrown (ArrayIndexOutOfBoundException...) is considered as a
			// corrupted .fo stream
			throw new InvalidFormatException(InvalidFormatException.CORRUPTED_FO_STREAM, e.toString());
		}
	}

	private void sectionHeadersLoaded(SectionHeader[] sectionHeaders) throws InvalidFormatException {
		// First pass on sections to compute required ROM & RAM total sizes and sections
		// relative addresses.
		int nbSections = sectionHeaders.length;
		int[] sectionsRelativeAdresses = new int[nbSections];
		this.sectionsRelativeAdresses = sectionsRelativeAdresses;
		boolean[] hasRelocations = new boolean[nbSections];
		this.crcs = new int[nbSections];

		int headerSize = DynamicLoaderNatives.getInstalledFeatureStructSize(NB_ROM_SECTIONS_WITH_CHECKSUM);
		int nextAddressROM = headerSize;
		int nextAddressRAM = 0;

		for (int sectionIndex = 1; sectionIndex < nbSections; ++sectionIndex) {
			SectionHeader sectionHeader = sectionHeaders[sectionIndex];
			if ((sectionHeader.flags & SHF_ALLOC) == SHF_ALLOC) {
				if (sectionHeader.type == SHT_PROGBITS) {
					nextAddressROM = align(nextAddressROM, sectionHeader.addralign);
					sectionsRelativeAdresses[sectionIndex] = nextAddressROM;
					nextAddressROM = nextAddressROM + sectionHeader.size;
				} else if (sectionHeader.type == SHT_NOBITS) {
					nextAddressRAM = align(nextAddressRAM, sectionHeader.addralign);
					sectionsRelativeAdresses[sectionIndex] = nextAddressRAM;
					nextAddressRAM = nextAddressRAM + sectionHeader.size;
				}
			}
			int relocatedSectionIndex = getRelocatedSection(sectionHeader);
			if (relocatedSectionIndex != 0) {
				hasRelocations[relocatedSectionIndex] = true;
			}
		}
		int sizeROM = nextAddressROM;
		int sizeRAM = nextAddressRAM;

		// - Allocate Feature ROM & RAM areas
		int featureHandle = DynamicLoaderNatives.allocateFeature(sizeROM, sizeRAM);
		if (featureHandle == Feature.FEATURE_HANDLE_NONE) {
			throw new InvalidFormatException(InvalidFormatConstants.FEATURE_ALLOCATION_ERROR);
		} else {
			this.featureHandle = featureHandle;
			this.startAddressROM = checkFeatureAddress(DynamicLoaderNatives.getFeatureAddressROM(featureHandle));
			this.startAddressRAM = checkFeatureAddress(DynamicLoaderNatives.getFeatureAddressRAM(featureHandle));
			this.sizeROM = sizeROM;
			this.sizeRAM = sizeRAM;
			this.headerSize = headerSize;
		}
	}

	private ProgAllocationSection newProgAllocationSection(InputStream is, int sectionIndex,
			SectionHeader sectionHeader) throws InvalidFormatException, IOException {
		assert this.currentRelocatedCodeSectionIndex == SHN_UNDEF;
		assert this.featureHandle != Feature.FEATURE_HANDLE_NONE;

		int size = sectionHeader.size;

		MemoryBlock allocatedBlock;
		boolean startCRC;
		boolean endCRC;
		MemoryBlock relocatedBlock = new ROMBlock(this.startAddressROM + getSectionRelativeAddress(sectionIndex), size);
		if (sectionIndex == RESOURCES_SECTION_INDEX) {
			// Resource section with no relocations. The content is directly transfered
			// to the relocatedBlock.
			allocatedBlock = relocatedBlock;
			startCRC = true;
			endCRC = true;
		} else {
			// Code section with relocations: the content is copied to RAM.
			// Then it will be transfered to ROM location after applying relocations.
			allocatedBlock = getFeatureCodeChunkBlock(size);

			if (sectionIndex == FIRST_CODE_SECTION_INDEX) {
				// Start the CRC computation for the future transfers to ROM.
				// The CRC will be finalized once all sections are processed.
				startCRC = true;
				endCRC = false;
			} else {
				// Continue with a next code chunk: update the offset to the beginning of the code section
				assert sectionHeader.link == FIRST_CODE_SECTION_INDEX;
				startCRC = false;
				endCRC = false;
			}
			this.currentRelocatedCodeSectionIndex = sectionIndex;
		}

		ProgAllocationSection section = new ProgAllocationSection(sectionIndex, size, allocatedBlock, relocatedBlock);
		if (startCRC) {
			startCRC();
		}
		transferSectionDataToMemoryBlock(section, is);
		if (endCRC) {
			endCRC(section);
		}
		return section;
	}

	private RAMBlock getFeatureCodeChunkBlock(int size) throws InvalidFormatException {
		RAMBlock allocatedBlock = this.featureCodeChunkBlock;
		if (size > allocatedBlock.getSize()) {
			throw new InvalidFormatException(FEATURE_CODE_CHUNK_TOO_SMALL);
		}
		return allocatedBlock;
	}

	private void transferCurrentProgAllocationSection() throws InvalidFormatException {
		int currentRelocatedCodeSectionIndex = this.currentRelocatedCodeSectionIndex;
		if (currentRelocatedCodeSectionIndex != SHN_UNDEF) {
			// Transfer the section
			AllocationSection relocatedSection = getExpectedAllocationSection(currentRelocatedCodeSectionIndex);
			long sectionChunkStartAddress = relocatedSection.getRelocatedAddress();
			MemoryBlock memoryBlock = relocatedSection.getMemoryBlock();
			long address = memoryBlock.getAddress();
			if (address != sectionChunkStartAddress) {
				assert memoryBlock instanceof RAMBlock;
				assert this.featureHandle != Feature.FEATURE_HANDLE_NONE;
				int size = relocatedSection.memSize();
				copyToROM(sectionChunkStartAddress, (RAMBlock) memoryBlock, size);
			} else {
				// the content has already been transfered to the relocated block during
				// ELF loading (resources section with no relocations)
			}

			this.currentRelocatedCodeSectionIndex = SHN_UNDEF;
		}
	}

	private int getSectionRelativeAddress(int sectionIndex) {
		int[] sectionsRelativeAdresses = this.sectionsRelativeAdresses;
		assert sectionsRelativeAdresses != null;
		return sectionsRelativeAdresses[sectionIndex];
	}

	private DynamicAllocationSection newDynamicAllocationSection(int sectionIndex, SectionHeader sectionHeader) {
		assert this.featureHandle != Feature.FEATURE_HANDLE_NONE;
		int size = sectionHeader.size;
		MemoryBlock allocatedBlock = new ROMBlock(this.startAddressRAM + getSectionRelativeAddress(sectionIndex), size);
		return new DynamicAllocationSection(allocatedBlock);
	}

	private void onCopyError() throws InvalidFormatException {
		throw newInvalidFormatCopyError();
	}

	private int getTransferBufferSize() {
		return Constants.getInt(TMP_TRANSFER_BUFFER_SIZE_CONSTANT);
	}

	/**
	 * Get the next aligned value.
	 *
	 * @param value
	 *            the value to be aligne
	 * @param alignment
	 *            the requested alignment (power of 2)
	 * @return the next aligned value.
	 */
	public static int align(int value, int alignment) {
		assert alignment > 0 && isPowerOfTwo(alignment);
		int alignmentMinusOne = alignment - 1;
		return (value + alignmentMinusOne) & ~alignmentMinusOne;
	}

	private static boolean isPowerOfTwo(int align) {
		// power of two: at most one bit set
		for (int i = 32; --i >= 0;) {
			if ((align & 1) == 1) {
				return (align & 0xFFFFFFFE) == 0;
			}
			align >>>= 1;
		}
		return false;
	}

	/**
	 * Require {@link #loadFO(byte[], InputStream)} called before.
	 *
	 * @return the allocated Feature handle.
	 */
	public int getFeatureHandle() {
		return this.featureHandle;
	}

	private void transferSectionDataToMemoryBlock(ProgAllocationSection section, InputStream is)
			throws IOException, InvalidFormatException {
		// Extract the data of the section described by the given section header
		byte[] bytes = new byte[getTransferBufferSize()];
		int size = section.memSize();
		MemoryBlock block = section.getMemoryBlock();
		int totalNbRead = 0;
		while (totalNbRead < size) {
			int nbRead = Math.min(size - totalNbRead, bytes.length);
			readFully(is, bytes, 0, nbRead);
			boolean copyOK = block.copyByteArrayToMemory(bytes, 0, totalNbRead, nbRead);
			if (!copyOK) {
				onCopyError();
			}
			totalNbRead += nbRead;
		}
	}

	private void startCRC() {
		DynamicLoaderNatives.copyToROMCRCStart();
	}

	private void endCRC(ProgAllocationSection section) {
		int[] crcs = this.crcs;
		assert crcs != null;
		crcs[section.getSectionIndex()] = DynamicLoaderNatives.copyToROMCRCEnd();
	}

	private int getCRC(int sectionIndex) {
		int[] crcs = this.crcs;
		assert crcs != null;
		return crcs[sectionIndex];
	}

	private void copyToROM(long relocatedAddress, RAMBlock block, int size) throws InvalidFormatException {
		checkCopyToROM(DynamicLoaderNatives.copyRAMToROM(relocatedAddress, block.memoryBlockDescriptor, 0, size));
	}

	private static void checkCopyToROM(boolean result) throws InvalidFormatException {
		if (!result) {
			throw newInvalidFormatCopyError();
		}
	}

	private static long checkFeatureAddress(long startAddressROMOrRAM) throws InvalidFormatException {
		// DynamicLoaderNatives.getFeatureAddressROM(int) and DynamicLoaderNatives.getFeatureAddressRAM return 0 if the
		// address given by the LL API does not respect the memory alignment.
		if (startAddressROMOrRAM == 0) {
			throw new InvalidFormatException(INVALID_AREA_START_ADDRESS_ALIGNMENT);
		}
		return startAddressROMOrRAM;
	}

	private static InvalidFormatException newInvalidFormatCopyError() {
		return new InvalidFormatException(UNEXPECTED_COPY_ERROR);
	}

	/**
	 * Gets the section index targetted by the given relocation section.
	 *
	 * @param sh
	 *            the section header
	 * @return the section index or <code>0</code> if the given section header is not a relocation section
	 */
	private static int getRelocatedSection(SectionHeader sh) {
		int type = sh.type;
		int targetSectionIndex;
		if (type == SHT_REL || type == SHT_RELA) {
			targetSectionIndex = sh.info;
			assert targetSectionIndex != 0;
		} else if (type == SECTION_TYPE_REL_SOAR) {
			targetSectionIndex = sh.link;
			assert targetSectionIndex != 0;
		} else {
			targetSectionIndex = 0;
		}
		return targetSectionIndex;
	}

	/**
	 * Get the non null section at the given index.
	 *
	 * @param index
	 *            the ELF section index
	 *
	 * @return the non null section at the given index
	 */
	private Section getExpectedSection(int index) {
		SectionHeader[] sectionHeaders = this.sectionHeaders;
		assert sectionHeaders != null;
		Section s = sectionHeaders[index].section;
		assert s != null;
		return s;
	}

	private AllocationSection getExpectedAllocationSection(int index) {
		return (AllocationSection) getExpectedSection(index);
	}

	private void load(byte[] elfSignature, InputStream is)
			throws IOException, IncompatibleFeatureException, InvalidFormatException {
		assert canLoad(elfSignature);

		// Load ELF header
		Header elfHeader = loadELFUnitHeader(is);

		// Load sections header
		SectionHeader[] sectionHeaders = loadSectionHeaders(elfHeader, is);
		this.sectionHeaders = sectionHeaders;

		// Process sections headers and resolve sections absolute addresses
		sectionHeadersLoaded(sectionHeaders);

		int featureHandle = this.featureHandle;
		assert featureHandle != Feature.FEATURE_HANDLE_NONE;

		// Process sections
		int nbSections = sectionHeaders.length;
		for (int i = 0; ++i < nbSections;) { // skip first section header
			SectionHeader sh = sectionHeaders[i];
			assert sh != null;
			sh.section = loadAndProcessSection(elfHeader, i, sh, is);
		}
		// transfer last section
		transferCurrentProgAllocationSection();

		// find root symbol
		int featuresHeaderStart = getSymbolAddress(FEATURES_START_SYMBOL);

		// Finalize the CRC of the code section
		endCRC((ProgAllocationSection) getExpectedSection(FIRST_CODE_SECTION_INDEX));

		// Initialize the ROM area header
		int headerSize = this.headerSize;

		RAMBlock headerBlock = getFeatureCodeChunkBlock(headerSize);
		DynamicLoaderNatives.initializeInstalledFeatureStruct(headerBlock.memoryBlockDescriptor, featureHandle,
				featuresHeaderStart, this.sizeROM, this.sizeRAM, NB_ROM_SECTIONS_WITH_CHECKSUM);

		initCRC(headerBlock, RESOURCES_SECTION_INDEX, 0);
		initCRC(headerBlock, FIRST_CODE_SECTION_INDEX, 1);
		DynamicLoaderNatives.finalizeInstalledFeatureStruct(headerBlock.memoryBlockDescriptor);

		copyToROM(this.startAddressROM, headerBlock, headerSize);

		// The last call to notify all bytes have been transfered to ROM
		// This may take a long time, especially in case of In System Programming to Flash Memory.
		checkCopyToROM(DynamicLoaderNatives.flushCopyToROM());

		// Here, the Feature is fully linked in memory and can be installed to the Core Engine.
	}

	private void initCRC(RAMBlock headerBlock, int sectionIndex, int crcROMSectionIndex) {
		int relativeAddressROM = getSectionRelativeAddress(sectionIndex);
		int size;
		switch (sectionIndex) {
		case RESOURCES_SECTION_INDEX: {
			size = getExpectedAllocationSection(sectionIndex).memSize();
			break;
		}
		case FIRST_CODE_SECTION_INDEX: {
			size = this.sizeROM - relativeAddressROM;
			break;
		}
		default:
			throw new AssertionError();
		}
		assert size >= 0;
		int crc = getCRC(sectionIndex);
		DynamicLoaderNatives.initializeInstalledFeatureStructCRCSection(headerBlock.memoryBlockDescriptor,
				crcROMSectionIndex, relativeAddressROM, size, crc);
	}

	/**
	 * Load and process a section. A {@link Section} instance is returned if and only if the section must be kept.
	 * {@link #absoluteBytesPtr} is updated.
	 *
	 * @param elfHeader
	 *            the ELF header
	 *
	 * @param sectionIndex
	 *            the section header index to process
	 * @param sh
	 *            the section header
	 *
	 * @param is
	 *            the <code>.fo</code> input stream
	 * @return a {@link Section} instance of <code>null</code> if the section is fully processed and does not need to be
	 *         kept in memory.
	 * @throws IOException
	 *             on I/O errors
	 * @throws InvalidFormatException
	 *             if the Feature file cannot be loaded
	 * @throws IncompatibleFeatureException
	 *             if the Feature is not compatible with this Kernel
	 */
	@Nullable
	private Section loadAndProcessSection(Header elfHeader, int sectionIndex, SectionHeader sh, InputStream is)
			throws IOException, InvalidFormatException, IncompatibleFeatureException {
		// skip padding bytes
		int padding = sh.offset - this.absoluteBytesPtr;
		if (padding < 0) {
			throw new InvalidFormatException(INVALID_FO_NEGATIVE_PADDING);
		} else if (padding > 0) {
			skipFully(is, padding);
		}

		int flags = sh.flags;
		switch (sh.type) {
		case SHT_SYMTAB: {
			SymbolTableSection s = new SymbolTableSection(extractDataAsByteArray(sh, is));
			StringSection stringTable = (StringSection) getExpectedSection(sh.link);
			s.stringSection = stringTable;

			if (sh.entsize != SYMBOL_TABLE_ENTRY_SIZE) {
				throw new InvalidFormatException(INVALID_FO_WRONG_SYMBOL_TABLE_SECTION_SIZE);
			}

			if (this.symbolTableSection != null) {
				throw new InvalidFormatException(INVALID_FO_WRONG_SYMBOL_TABLE_SECTION_COUNT);
			}
			this.symbolTableSection = s;

			return s;
		}
		case SHT_PROGBITS: {
			if ((flags & SHF_ALLOC) == SHF_ALLOC) {
				// Here, found a new ProgBits section: transfer the current one (relocations have been applied)
				transferCurrentProgAllocationSection();
				return newProgAllocationSection(is, sectionIndex, sh);
			} else {
				// control section: data not loaded
				skipFully(is, sh.size);
			}
			return null;
		}
		case SHT_STRTAB: {
			if (sectionIndex == elfHeader.e_shstrndx) {
				// Reaching section names : data not loaded
				skipFully(is, sh.size);

				// Here, we are just before starting applying relocations
				if (this.symbolTableSection == null) {
					throw new InvalidFormatException(INVALID_FO_WRONG_SYMBOL_TABLE_SECTION_MISSING);
				}
				this.typeHeaderAddress = getSymbolAddress(TYPES_HEADER_SYMBOL);
				return null;
			} else {
				byte[] data = extractDataAsByteArray(sh, is);
				if (sectionIndex == KERNEL_UID_SECTION_INDEX) {
					// Check this Feature can be linked on this Kernel.
					if (!KernelNatives.checkKernel(data, this.lsbAndNotMsb)) {
						throw new IncompatibleFeatureException(IncompatibleFeatureException.WRONG_KERNEL_UID);
					}
					return null;
				} else {
					return new StringSection(data);
				}
			}
		}
		case SHT_REL:
		case SHT_RELA: {
			// Skip FO file with metadata section.
			// ('.rel.debug.soar section' of <code>.fodbg</code> file)
			skipFully(is, sh.size);
			return null;
		}
		case SHT_NOBITS: {
			if ((flags & SHF_ALLOC) == SHF_ALLOC) {
				return newDynamicAllocationSection(sectionIndex, sh);
			} else {
				throw new InvalidFormatException(INVALID_FO_WRONG_NOBITS_SECTION_FLAG);
			}
		}
		case SECTION_TYPE_REL_SOAR: {
			assert sh.link == FIRST_CODE_SECTION_INDEX;
			byte[] bytes = extractDataAsByteArray(sh, is);
			RelocationSectionSoar rs = new RelocationSectionSoar(bytes);
			applyRelocations(rs);
			return null;
		}
		default:
			throw new InvalidFormatException(INVALID_FO_WRONG_UNKNOWN_SECTION_TYPE);
		}
	}

	/**
	 * Loads the ELF header starting right after the ELF magic field (the first 4 bytes have been read).
	 *
	 * @param is
	 *            the input stream
	 * @return the ELF header
	 * @throws IOException
	 *             on I/O errors
	 * @throws InvalidFormatException
	 *             if the Feature file cannot be loaded
	 * @throws IncompatibleFeatureException
	 *             if the Feature is not compatible with this Kernel
	 */
	private Header loadELFUnitHeader(InputStream is)
			throws IOException, InvalidFormatException, IncompatibleFeatureException {
		Header header = new Header();
		this.absoluteBytesPtr = SIGNATURE_LENGTH;

		byte[] bytes = new byte[HEADER_SIZE];
		readFully(is, bytes, SIGNATURE_LENGTH, bytes.length - SIGNATURE_LENGTH);

		header.e_ident_class = bytes[EI_CLASS];
		if (header.e_ident_class != ELFCLASS32) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_CLASS);
		}

		header.e_ident_data = bytes[EI_DATA];
		if (header.e_ident_data != ELFDATA2LSB && header.e_ident_data != ELFDATA2MSB) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_ENDIANNESS);
		}
		this.lsbAndNotMsb = (header.e_ident_data == ELFDATA2LSB);

		header.e_ident_version = bytes[EI_VERSION];
		if (header.e_ident_version != EV_CURRENT) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_VERSION);
		}

		header.e_ident_osabi = bytes[EI_OSABI];

		header.e_ident_abiversion = bytes[EI_ABIVERSION];
		if (header.e_ident_abiversion != SUPPORTED_ABI_VERSION) {
			throw new IncompatibleFeatureException(IncompatibleFeatureException.WRONG_FO_VERSION);
		}

		int ptr = EI_NIDENT;
		header.e_type = u2(bytes, ptr);
		ptr += 2;

		if (header.e_type != ET_REL) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_TYPE);
		}

		header.e_machine = u2(bytes, ptr);
		ptr += 2;

		// Check ELF FO file machine
		if (header.e_machine != EM_SOARMICROJVM) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_MACHINE);
		}

		header.e_version = u4(bytes, ptr);
		ptr += 4;

		if (header.e_version != EV_CURRENT) {
			throw new InvalidFormatException(INVALID_FO_WRONG_ELF_VERSION);
		}

		header.e_entry = u4(bytes, ptr);
		ptr += 4;
		header.e_phoff = u4(bytes, ptr);
		ptr += 4;
		header.e_shoff = u4(bytes, ptr);

		// Check ELF FO file section header appears right after the program header
		if (header.e_shoff != HEADER_SIZE) {
			throw new InvalidFormatException(INVALID_FO_WRONG_SECTION_HEADER);
		}

		ptr += 4;
		header.e_flags = u4(bytes, ptr);
		ptr += 4;
		header.e_ehsize = u2(bytes, ptr);
		ptr += 2;
		header.e_phentsize = u2(bytes, ptr);
		ptr += 2;
		header.e_phnum = u2(bytes, ptr);
		ptr += 2;
		header.e_shentsize = u2(bytes, ptr);
		ptr += 2;
		header.e_shnum = u2(bytes, ptr);

		if (header.e_shnum == SHN_UNDEF) {
			throw new InvalidFormatException(INVALID_FO_WRONG_SECTION_NUM);
		}

		ptr += 2;
		header.e_shstrndx = u2(bytes, ptr);

		return header;
	}

	private SectionHeader[] loadSectionHeaders(Header elfHeader, InputStream is)
			throws IOException, InvalidFormatException {
		// load sections headers
		// At least one section header at index 0 that represent no section
		assert this.absoluteBytesPtr == HEADER_SIZE;
		int size = SECTION_HEADER_SIZE;
		byte[] bytes = new byte[size];
		readFully(is, bytes, 0, bytes.length);

		SectionHeader firstSection = new SectionHeader();
		loadHeader(firstSection, bytes, 0);

		int nbSections = elfHeader.e_shnum;

		int strTableHeaderIndex = elfHeader.e_shstrndx;
		if (strTableHeaderIndex == SHN_XINDEX) {
			strTableHeaderIndex = firstSection.link;
		}

		size = SECTION_HEADER_SIZE * (nbSections - 1);
		bytes = new byte[size];
		readFully(is, bytes, 0, bytes.length);

		SectionHeader strTableHeader = null;
		SectionHeader[] sectionHeaders = new SectionHeader[nbSections];
		sectionHeaders[0] = firstSection;
		int ptr = 0;
		for (int i = 0; ++i < nbSections;) {
			SectionHeader sh = sectionHeaders[i] = new SectionHeader();
			if (i == strTableHeaderIndex) {
				strTableHeader = sh;
			}
			ptr = loadHeader(sh, bytes, ptr);
		}

		if (strTableHeader == null) {
			throw new InvalidFormatException(INVALID_FO_WRONG_SECTION_NAMES);
		}

		return sectionHeaders;
	}

	private void readFully(InputStream is, byte[] bytes, int offset, int length) throws IOException {
		int end = offset + length;
		while (offset < end) {
			int nbRead = is.read(bytes, offset, end - offset);
			if (nbRead == -1) {
				throw new EOFException();
			}
			offset += nbRead;
		}
		this.absoluteBytesPtr += length;
	}

	private void skipFully(InputStream is, int length) throws IOException {
		this.absoluteBytesPtr += length;
		while (length > 0) {
			long nbSkipped = is.skip(length);
			if (nbSkipped == -1) {
				throw new EOFException();
			}
			length -= nbSkipped;
		}
	}

	private int u2(byte[] bytes, int ptr) {
		return u2(bytes, ptr, this.lsbAndNotMsb);
	}

	private int u4(byte[] bytes, int ptr) {
		return u4(bytes, ptr, this.lsbAndNotMsb);
	}

	private static int u2(byte[] bytes, int ptr, boolean lsbAndNotMsb) {
		if (lsbAndNotMsb) {
			return (bytes[ptr++] & 0xFF) | ((bytes[ptr] & 0xFF) << 8);
		} else {
			return ((bytes[ptr++] & 0xFF) << 8) | (bytes[ptr] & 0xFF);
		}
	}

	private static int u4(byte[] bytes, int ptr, boolean lsbAndNotMsb) {
		if (lsbAndNotMsb) {
			return (bytes[ptr++] & 0xFF) | ((bytes[ptr++] & 0xFF) << 8) | ((bytes[ptr++] & 0xFF) << 16)
					| ((bytes[ptr] & 0xFF) << 24);
		} else {
			return ((bytes[ptr++] & 0xFF) << 24) | ((bytes[ptr++] & 0xFF) << 16) | ((bytes[ptr++] & 0xFF) << 8)
					| (bytes[ptr] & 0xFF);
		}
	}

	/**
	 * Tells whether the given bytes indicate the ELF file signature.
	 *
	 * @param bytes
	 *            the first 4 bytes of the stream
	 * @return <code>true</code> if this is an ELF file, <code>false</code> otherwise.
	 */
	public static boolean canLoad(byte[] bytes) {
		if (bytes.length < SIGNATURE_LENGTH) {
			return false;
		}

		// read magic fields
		return (bytes[EI_MAG0] == ELFMAG0 && bytes[EI_MAG1] == ELFMAG1 && bytes[EI_MAG2] == ELFMAG2
				&& bytes[EI_MAG3] == ELFMAG3);
	}

	private void applyRelocations(RelocationSectionSoar rs) throws InvalidFormatException {
		assert this.currentRelocatedCodeSectionIndex != SHN_UNDEF;
		ProgAllocationSection relocatedSection = (ProgAllocationSection) getExpectedSection(
				this.currentRelocatedCodeSectionIndex);

		long sectionCodeStartAddress = getExpectedAllocationSection(FIRST_CODE_SECTION_INDEX).getRelocatedAddress();
		long sectionChunkStartAddress = relocatedSection.getRelocatedAddress();
		long sectionResourceStartAddress = getExpectedAllocationSection(RESOURCES_SECTION_INDEX).getRelocatedAddress();
		long sectionStaticsStartAddress = getExpectedAllocationSection(STATICS_SECTION_INDEX).getRelocatedAddress();

		MemoryBlock data = relocatedSection.getMemoryBlock();
		long memoryBlockDescriptor = ((RAMBlock) data).memoryBlockDescriptor;

		byte[] relocationData = rs.data;
		int relocationsCount = Constants.getInt(RELOC_CHUNK_CONSTANT);
		int startIndex = 0;
		while (true) {
			int res = DynamicLoaderNatives.applyRelocations(memoryBlockDescriptor, relocationData, startIndex,
					relocationsCount, this.typeHeaderAddress, sectionCodeStartAddress, sectionChunkStartAddress,
					sectionResourceStartAddress, sectionStaticsStartAddress);
			if (res == 0) {
				break; // all relocations successfully applied
			} else if (res == -1) {
				throw new AssertionError();
			} else if (res > 0) {
				throw new InvalidFormatException(res);
			} else {
				int previousIndex = startIndex;
				startIndex = -res;
				assert startIndex > previousIndex && startIndex < relocationData.length;

				// give the hand to other threads of same priority before applying a new chunk of relocations
				Thread.yield();
			}
		}
	}

	private int loadHeader(SectionHeader h, byte[] bytes, int ptr) {
		// section name (unused)
		u4(bytes, ptr);
		ptr += 4;
		h.type = u4(bytes, ptr);
		ptr += 4;
		h.flags = u4(bytes, ptr);
		ptr += 4;
		// section address (unused)
		u4(bytes, ptr);
		ptr += 4;
		h.offset = u4(bytes, ptr);
		ptr += 4;
		h.size = u4(bytes, ptr);
		ptr += 4;
		h.link = u4(bytes, ptr);
		ptr += 4;
		h.info = u4(bytes, ptr);
		ptr += 4;
		h.addralign = u4(bytes, ptr);
		ptr += 4;
		h.entsize = u4(bytes, ptr);
		ptr += 4;
		return ptr;
	}

	/**
	 * Load a {@link SymbolTableEntry} from the {@link SymbolTableSection}
	 *
	 * @param section
	 *            the {@link SymbolTableSection}
	 * @param symbolIndex
	 *            the symbol index (0 based)
	 *
	 * @return the loaded symbol or <code>null</code> if symbolIndex is <code>0</code>
	 */
	@Nullable
	private SymbolTableEntry loadSymbolTableEntry(SymbolTableSection section, int symbolIndex) {
		if (symbolIndex == 0) {
			return null;
		}
		StringSection stringTable = section.stringSection;
		assert stringTable != null;
		boolean lsbAndNotMsb = this.lsbAndNotMsb;
		byte[] bytes = section.data;
		int ptr = symbolIndex * SYMBOL_TABLE_ENTRY_SIZE;
		int nameIndex = u4(bytes, ptr, lsbAndNotMsb);
		ptr += 4;
		byte[] name = stringTable.getName(nameIndex);
		int value = u4(bytes, ptr, lsbAndNotMsb);
		ptr += 4;
		ptr += 4;// skip size
		++ptr;// skip info
		++ptr;// skip other
		int sectionIndex = u2(bytes, ptr, lsbAndNotMsb);
		return new SymbolTableEntry(name, value, sectionIndex);
	}

	private byte[] extractDataAsByteArray(SectionHeader sh, InputStream is) throws IOException {
		// Extract the data of the section described by the given section header
		assert sh.type != SHT_NOBITS;
		byte[] data = new byte[sh.size];
		readFully(is, data, 0, data.length);
		return data;
	}

	/**
	 * Gets the address of the symbol with the given name
	 *
	 * @param symbolName
	 *            the symbol name
	 * @return the address of the symbol with the given name
	 * @throws InvalidFormatException
	 *             if the symbol cannot be found
	 * @see #getSymbolEntry(byte[])
	 */
	private int getSymbolAddress(byte[] symbolName) throws InvalidFormatException {
		return getSymbolAddress(getSymbolEntry(symbolName));
	}

	/**
	 * This is a linear search. Caller is strongly encouraged to store the result in a cache.
	 *
	 * @param symbolName
	 *            the symbol name
	 * @return the {@link SymbolTableEntry} corresponding to the given symbol name
	 * @throws InvalidFormatException
	 *             if the symbol cannot be found
	 */
	private SymbolTableEntry getSymbolEntry(byte[] symbolName) throws InvalidFormatException {
		// find the symbol entry
		SymbolTableSection section = this.symbolTableSection;
		assert section != null;
		int nbEntries = section.nbEntries();
		for (int i = nbEntries; --i > 0;) { // first symbol index is null
			SymbolTableEntry entry = loadSymbolTableEntry(section, i);
			assert entry != null;
			if (equals(entry.name, symbolName)) {
				return entry;
			}
		}
		throw new InvalidFormatException(INVALID_FO_UNKNOWN_SYMBOL, new String(symbolName));
	}

	private int getSymbolAddress(SymbolTableEntry symbolEntry) throws InvalidFormatException {
		if (symbolEntry.isUndefined()) {
			throw new InvalidFormatException(INVALID_FO_UNDEFINED_SYMBOL, new String(symbolEntry.name));
		} else if (symbolEntry.isAbsolute()) {
			return symbolEntry.value;
		} else {
			AllocationSection targetSection = getExpectedAllocationSection(symbolEntry.sectionIndex);
			return (int) (targetSection.getRelocatedAddress() + symbolEntry.value);
		}
	}

	public static boolean equals(byte[] array1, byte[] array2) {
		if (array1.length != array2.length) {
			return false;
		}
		for (int i = array1.length; --i >= 0;) { // reverse order : most likely symbols that have the same end are
			// equals
			if (array1[i] != array2[i]) {
				return false;
			}
		}
		return true;
	}
}
