/*
 * Java
 *
 * Copyright 2013-2024 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import java.io.IOException;
import java.io.InputStream;

import ej.annotation.Nullable;
import ej.kf.AlreadyLoadedFeatureException;
import ej.kf.Feature;
import ej.kf.IncompatibleFeatureException;
import ej.kf.InvalidFormatException;
import ej.kf.Kernel;

/**
 * The {@link IFeatureLoader} implementation for <code>.fo</code> files.
 */
public class FoFeatureLoader implements IFeatureLoader, InvalidFormatConstants {

	// NOTE:
	// All working fields required for installing a Feature must be stored in the FoLoader instance.
	// This instance is correctly freed when installation is done (both in case of success or failure).

	/**
	 * Has a mean only within {@link #load(byte[], InputStream)} process.
	 */
	@Nullable
	protected FoLoader foLoader;

	/**
	 * The default constructor.
	 */
	public FoFeatureLoader() {
		// Loader is dynamically created.
		// The property 'com.is2t.kf.dynamic.loader' must be updated if this class is renamed.
	}

	@Override
	public boolean canLoad(byte[] signature) {
		return FoLoader.canLoad(signature);
	}

	@Override
	public boolean canUnload() {
		return true;
	}

	@Override
	public Feature load(byte[] signature, InputStream is)
			throws IOException, InvalidFormatException, IncompatibleFeatureException, AlreadyLoadedFeatureException {
		// The ROM area is organized as following:
		// - Header (Installed Feature Struct + Kernel UID)
		// - PROGBITS resources section
		// - PROGBITS code section
		// The RAM area is organized as following:
		// - NOBITS section1
		// - ...
		// - NOBITS sectionN
		//
		// NOTE: The Kernel UID is the Kernel on which this Feature is linked, which may
		// not be the same than the Kernel UID embedded in the Feature (the
		// Kernel on which the Feature has been built).

		FoLoader foLoader = this.foLoader = new FoLoader();
		int featureHandle;
		Feature feature = null;
		try {
			// All memory allocated in this try-block will be released
			// if an exception is thrown.

			// - Load ELF unit
			foLoader.loadFO(signature, is);

			// add the linked Feature with no LLKERNEL callback in case of error
			feature = addLinkedFeature(foLoader.getFeatureHandle(), true);
			assert feature != null;
		} finally {
			featureHandle = foLoader.getFeatureHandle();
			if (feature == null && featureHandle != Feature.FEATURE_HANDLE_NONE) {
				// An error occurred and the feature has been allocated.
				// Free the allocated Feature
				DynamicLoaderNatives.freeFeature(featureHandle);
			}
			// Free memory working fields
			this.foLoader = null;
		}

		assert feature != null;
		return feature;
	}

	/**
	 * Check that the given Feature has been linked for the current Kernel.
	 *
	 * @param featureHandle
	 *            the feature handle, or {@link Feature#FEATURE_HANDLE_NONE}
	 * @return <code>true</code> if the Feature has been linked for the current Kernel, <code>false<code> if LLKERNEL
	 *         callback returned to silently ignore the error.
	 *
	 * @throws InvalidFormatException
	 *             if the Feature file cannot be loaded (only if LLKERNEL callback returned to throw the error)
	 * @throws IncompatibleFeatureException
	 *             if the Feature is not compatible with this Kernel (only if LLKERNEL callback returned to throw the
	 *             error)
	 */
	protected static boolean checkKernel(int featureHandle)
			throws InvalidFormatException, IncompatibleFeatureException {
		// Check Feature
		int checkKernelResult = DynamicLoaderNatives.checkKernel(featureHandle);
		switch (checkKernelResult) {
		case DynamicLoaderNatives.CHECK_KERNEL_IGNORE: {
			return false;
		}
		case DynamicLoaderNatives.CHECK_KERNEL_OK: {
			return true;
		}
		case DynamicLoaderNatives.CHECK_KERNEL_CORRUPTED_CONTENT: {
			throw new InvalidFormatException(CORRUPTED_INSTALLED_FEATURE);
		}
		case DynamicLoaderNatives.CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_UID: {
			throw new IncompatibleFeatureException(IncompatibleFeatureException.WRONG_KERNEL_UID);
		}
		case DynamicLoaderNatives.CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_ADDRESSES: {
			throw new IncompatibleFeatureException(IncompatibleFeatureException.WRONG_KERNEL_ADDRESSES);
		}
		case DynamicLoaderNatives.CHECK_KERNEL_RAM_ADDRESS_CHANGED: {
			throw new InvalidFormatException(RAM_ADDRESS_CHANGED);
		}
		case DynamicLoaderNatives.CHECK_KERNEL_NULL_HANDLE: {
			throw new InvalidFormatException(FEATURE_NULL_HANDLE);
		}
		default: {
			throw new AssertionError();
		}
		}
	}

	/**
	 * Add a linked Feature to the Kernel. Caller is responsible to ensure that the Feature has been linked for this
	 * Kernel. When this method is successful, it returns a new {@link Feature} object ready to be started.
	 *
	 * @param featureHandle
	 *            the feature handle
	 * @param noCallback
	 *            <code>true</code> if errors must be silently ignored, <code>false</code>
	 * @return the linked Feature or <code>null</code> if LLKERNEL callback returned to silently ignore the error
	 * @throws AlreadyLoadedFeatureException
	 *             if a Feature with the same UID is already loaded (if LLKERNEL callback returned to silently ignore
	 *             the error)
	 * @throws InvalidFormatException
	 *             if the Feature file cannot be loaded (if LLKERNEL callback returned to silently ignore the error)
	 * @throws IncompatibleFeatureException
	 *             if the Feature is not compatible with this Kernel (if LLKERNEL callback returned to silently ignore
	 *             the error)
	 */
	@Nullable
	private Feature addLinkedFeature(int featureHandle, boolean noCallback)
			throws AlreadyLoadedFeatureException, InvalidFormatException, IncompatibleFeatureException {
		assert featureHandle != Feature.FEATURE_HANDLE_NONE;

		// Register the Feature to dynamically installed Features
		long result = DynamicLoaderNatives.addLinkedFeature(featureHandle, noCallback);

		if (result >= DynamicLoaderNatives.LINK_FEATURE_ERROR_CODE_MIN
				&& result <= DynamicLoaderNatives.LINK_FEATURE_ERROR_CODE_MAX) {
			switch ((int) result) {
			case DynamicLoaderNatives.LINK_FEATURE_TOO_MANY_INSTALLED: {
				throw new InvalidFormatException(TOO_MANY_INSTALLED_FEATURES);
			}
			case DynamicLoaderNatives.LINK_FEATURE_ALREADY_INSTALLED: {
				long previousFeatureSectionDescriptor = DynamicLoaderNatives.getAlreadyInstalledFeatureDescriptor();
				Feature f = Kernel.getFeatureByDescriptor(previousFeatureSectionDescriptor);
				assert f != null;
				throw new AlreadyLoadedFeatureException(f);
			}
			case DynamicLoaderNatives.LINK_FEATURE_ROM_OVERLAP: {
				throw new InvalidFormatException(OVERLAP_ROM_AREA);
			}
			case DynamicLoaderNatives.LINK_FEATURE_RAM_OVERLAP: {
				throw new InvalidFormatException(OVERLAP_RAM_AREA);
			}
			case DynamicLoaderNatives.LINK_FEATURE_IGNORE: {
				return null;
			}
			default: {
				throw new AssertionError();
			}
			}
		} else {
			long featureSectionDescriptor = result;
			return new Feature(this, featureSectionDescriptor, featureHandle);
		}
	}

	@Override
	public void loadInstalledFeatures()
			throws InvalidFormatException, IncompatibleFeatureException, AlreadyLoadedFeatureException {
		// 1- Get all handles atomically
		int nbInstalledFeatures = DynamicLoaderNatives.getInstalledFeaturesCount();
		int[] handles = new int[nbInstalledFeatures];
		for (int i = 0; i < nbInstalledFeatures; ++i) {
			handles[i] = DynamicLoaderNatives.getInstalledFeatureHandle(i);
		}

		// 2- Try to add Installed Features.
		// On error, the LLKERNEL callback is called. Depending on the return code, an exception is thrown or the
		// Installed Feature is silently ignored.
		for (int i = 0; i < nbInstalledFeatures; ++i) {
			int featureHandle = handles[i];
			if (checkKernel(featureHandle)) {
				Feature f = addLinkedFeature(featureHandle, false);
				if (f != null) {
					Kernel.addInstalledFeature(f);
				}
			}
		}
	}

	@Override
	public boolean unload(Feature f) {
		boolean result = DynamicLoaderNatives.unlink(f);
		if (!result) {
			throw new AssertionError(); // sanity check
		}
		return true;
	}
}
