/*
 * Java
 *
 * Copyright 2013-2023 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.kf;

import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.InvalidFormatException;

public class DynamicLoaderNatives {

	public static final int CHECK_KERNEL_OK = 0; // VM KNOWN
	public static final int CHECK_KERNEL_IGNORE = 1; // VM KNOWN
	public static final int CHECK_KERNEL_CORRUPTED_CONTENT = 2;// VM KNOWN
	public static final int CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_UID = 3;// VM KNOWN
	public static final int CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_ADDRESSES = 4;// VM KNOWN
	public static final int CHECK_KERNEL_RAM_ADDRESS_CHANGED = 5;// VM KNOWN
	public static final int CHECK_KERNEL_NULL_HANDLE = 6;// VM KNOWN

	public static final int LINK_FEATURE_IGNORE = -1;// VM KNOWN
	public static final int LINK_FEATURE_TOO_MANY_INSTALLED = -2;// VM KNOWN
	public static final int LINK_FEATURE_ALREADY_INSTALLED = -3;// VM KNOWN
	public static final int LINK_FEATURE_ROM_OVERLAP = -4;// VM KNOWN
	public static final int LINK_FEATURE_RAM_OVERLAP = -5;// VM KNOWN

	public static final int LINK_FEATURE_ERROR_CODE_MIN = LINK_FEATURE_RAM_OVERLAP;
	public static final int LINK_FEATURE_ERROR_CODE_MAX = LINK_FEATURE_IGNORE;

	/**
	 * Allocates RAM and ROM areas for the Feature being loaded.
	 *
	 * @param sizeROM
	 *            the required ROM size in bytes
	 * @param sizeRAM
	 *            the required RAM size in bytes
	 * @return the Feature handle or <code>0</code> if Feature cannot be allocated
	 */
	public static native int allocateFeature(int sizeROM, int sizeRAM);

	/**
	 * This native is only called when something goes wrong during Feature installation, until it is not linked in the
	 * VM. On regular uninstall, the {@link #unlink(Feature)} method is called.
	 *
	 * @param featureHandle
	 *            a non null feature handle
	 */
	public static native void freeFeature(int featureHandle);

	/**
	 * Gets the start address of ROM area
	 *
	 * @param featureHandle
	 *            a non null feature handle
	 * @return the start address of ROM area, or <code>0</code> if the address does not match the required alignment
	 *         constraints
	 */
	public static native long getFeatureAddressROM(int featureHandle);

	/**
	 * Gets the start address of RAM area
	 *
	 * @param featureHandle
	 *            a non null feature handle
	 * @return the start address of RAM area, or <code>0</code> if the address does not match the required alignment
	 *         constraints
	 */
	public static native long getFeatureAddressRAM(int featureHandle);

	/**
	 * Get the start address of the given memory block descriptor
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 * @return the start address
	 */
	public static native long getMemoryBlockStartAddress(long memoryBlockDescriptor);

	/**
	 * Get the size of the given memory block descriptor
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 * @return the size in bytes
	 */
	public static native int getMemoryBlockSize(long memoryBlockDescriptor);

	/**
	 * Get the statically allocated RAM block for Feature code relocation.
	 *
	 * @return the memory block descriptor of the statically allocated RAM block for Feature code relocation.
	 */
	public static native long getFeatureCodeChunk();

	/**
	 * Copy bytes from the given bytes array to the chunk of memory.
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 * @param src
	 *            the source bytes array
	 * @param srcOffset
	 *            the offset in the source array
	 * @param dstOffset
	 *            the offset in the destination memory
	 * @param length
	 *            the number of bytes to copy
	 *
	 * @return <code>true</code> on success, <code>false</code> if the number of bytes is greater than the size of the
	 *         block
	 */
	public static native boolean copyByteArrayToRAM(long memoryBlockDescriptor, byte[] src, int srcOffset,
			int dstOffset, int length);

	/**
	 * Copy bytes from the chunk of memory to the given bytes array.
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 * @param srcOffset
	 *            the offset in the source memory
	 * @param dst
	 *            the destination bytes array
	 * @param dstOffset
	 *            the offset in the destination array
	 * @param length
	 *            the number of bytes to copy
	 *
	 * @return <code>true</code> on success, <code>false</code> if the number of bytes is greater than the size of the
	 *         block
	 */
	public static native boolean copyRAMToByteArray(long memoryBlockDescriptor, int srcOffset, byte[] dst,
			int dstOffset, int length);

	/**
	 * Copy bytes from the chunk of memory to ROM. This function does not require that all bytes are really copied to
	 * the destination before it returns. (see <code>LLKERNEL_IMPL_copyToROM</code>).
	 *
	 * @param destAddress
	 *            the destination address in ROM
	 * @param srcMemoryBlockDescriptor
	 *            the memory block descriptor
	 * @param srcOffset
	 *            the offset in the source memory
	 * @param size
	 *            the number of bytes to copy
	 *
	 * @return <code>true</code> on success, <code>false</code> on error
	 */
	public static native boolean copyRAMToROM(long destAddress, long srcMemoryBlockDescriptor, int srcOffset, int size);

	/**
	 * Copy array of bytes to ROM. This function does not require that all bytes are really copied to the destination
	 * before it returns.(see <code>LLKERNEL_IMPL_copyToROM</code>).
	 *
	 * @param destAddress
	 *            the destination address in ROM
	 * @param src
	 *            the source buffer
	 * @param offset
	 *            the offset of the first byte to copy
	 * @param size
	 *            the number of bytes to copy
	 *
	 * @return <code>true</code> on success, <code>false</code> on error
	 */
	public static native boolean copyByteArrayToROM(long destAddress, byte[] src, int offset, int size);

	/**
	 * This function must block until all bytes are really copied to ROM area.(see
	 * <code>LLKERNEL_IMPL_flushCopyToROM</code>).
	 *
	 * @return <code>true</code> on success, <code>false</code> on error
	 */
	public static native boolean flushCopyToROM();

	/**
	 * Add a linked Feature to the VM.
	 *
	 * @param featureHandle
	 *            the Feature handle
	 * @param noCallback
	 *            <code>true</code> if the LLKERNEL callback must not be called, <code>false</code> otherwise
	 * @return the vm descriptor on success, or an error code between {@link #LINK_FEATURE_ERROR_CODE_MIN} and
	 *         {@link #LINK_FEATURE_ERROR_CODE_MAX} on error
	 * @see #unlink(Feature)
	 */
	public static native long addLinkedFeature(int featureHandle, boolean noCallback);

	/**
	 * Get the descriptor of the already installed feature. This function can only be called after
	 * {@link #addLinkedFeature(int, boolean)} when it returned {@link #LINK_FEATURE_ALREADY_INSTALLED}.
	 *
	 * @return the descriptor of the already installed feature
	 */
	public static native long getAlreadyInstalledFeatureDescriptor();

	/**
	 * Unlink a Feature.
	 *
	 * @param feature
	 *            native assumes it is not null (no check)
	 * @return true on success, false if this Feature is not in the {@link State#INSTALLED} state.
	 */
	public static native boolean unlink(Feature feature);

	/**
	 * Apply relocations (SOARv2) in native side.
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block of the relocated section.
	 * @param relocationData
	 *            the relocation data (SOAR format).
	 * @param startIndex
	 *            the offset in bytes of the next relocation to apply.
	 * @param relocationsCount
	 *            the number of relocations to apply before returning.
	 * @param typeHeaderAddress
	 *            the start address of the type headers.
	 * @param sectionCodeStartAddress
	 *            the start address of the code section (ROM area).
	 * @param sectionCodeChunkStartAddress
	 *            the start address of this code chunk being relocated (ROM area).
	 * @param sectionResourceStartAddress
	 *            the start address of the resources section (ROM area).
	 * @param sectionStaticsStartAddress
	 *            the start address of the statics section (RAM area).
	 * @return the following codes:
	 *         <li><code>0</code> on success</li>
	 *         <li><code>-1</code> on internal assertion error</li>
	 *         <li>a negative value: the (negated) index of the next relocation to apply (when the relocations count is
	 *         reached)</li>
	 *         <li>a positive value: an {@link InvalidFormatException} specific error code</li>
	 */
	public static native int applyRelocations(long memoryBlockDescriptor, byte[] relocationData, int startIndex,
			int relocationsCount, long typeHeaderAddress, long sectionCodeStartAddress,
			long sectionCodeChunkStartAddress, long sectionResourceStartAddress, long sectionStaticsStartAddress);

	/**
	 * Get the size in bytes of the Installed Features header. ROM sections (PROGBITS + Flag A) will have an associated
	 * computed checksum.
	 *
	 * @param nbROMSections
	 *            the number of ROM sections
	 * @return the size in bytes of the header structure
	 */
	public static native int getInstalledFeatureStructSize(int nbROMSections);

	/**
	 * Initialize the header structure within the given memory block
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 * @param featureHandle
	 *            the feature handle
	 * @param featuresHeaderStart
	 *            the address of the Feature header structure
	 * @param sizeROM
	 *            the size in ROM
	 * @param sizeRAM
	 *            the size in RAM
	 * @param nbROMSections
	 *            the number of ROM sections
	 */
	public static native void initializeInstalledFeatureStruct(long memoryBlockDescriptor, int featureHandle,
			int featuresHeaderStart, int sizeROM, int sizeRAM, int nbROMSections);

	/**
	 * Initialize the CRC of a ROM section
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 *
	 * @param index
	 *            the 0 based CRC section index
	 * @param relativeAddressROM
	 *            the relative start address of the section in ROM
	 * @param size
	 *            the section size in bytes
	 * @param crc
	 *            the computed CRC
	 */
	public static native void initializeInstalledFeatureStructCRCSection(long memoryBlockDescriptor, int index,
			int relativeAddressROM, int size, int crc);

	/**
	 * Finalize the header structure within the given memory block
	 *
	 * @param memoryBlockDescriptor
	 *            the memory block descriptor
	 */
	public static native void finalizeInstalledFeatureStruct(long memoryBlockDescriptor);

	/**
	 * See <code>LLKERNEL_IMPL_getAllocatedFeaturesCount</code>
	 *
	 * @return the number of installed features
	 */
	public static native int getInstalledFeaturesCount();

	/**
	 * See <code>LLKERNEL_IMPL_getFeatureHandle</code>
	 *
	 * @param allocationIndex
	 *            a 0 based index
	 * @return the feature handle
	 */
	public static native int getInstalledFeatureHandle(int allocationIndex);

	/**
	 * Check if this Installed Feature can be initialized this Kernel.
	 *
	 * @param featureHandle
	 *            a feature handle or {@link Feature#FEATURE_HANDLE_NONE}.
	 *
	 * @return {@link #CHECK_KERNEL_OK} on success, {@link #CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_UID},
	 *         {@link #CHECK_KERNEL_INCOMPATIBLE_KERNEL_WRONG_ADDRESSES}, {@link #CHECK_KERNEL_CORRUPTED_CONTENT},
	 *         {@link #CHECK_KERNEL_RAM_ADDRESS_CHANGED}, {@link #CHECK_KERNEL_NULL_HANDLE} on failure, or
	 *         {@link #CHECK_KERNEL_IGNORE} if Feature initialization must be silently ignored
	 */
	public static native int checkKernel(int featureHandle);

	/**
	 * Starts a new CRC computation of next sequences of bytes that will be copied to ROM.
	 */
	public static native void copyToROMCRCStart();

	/**
	 * End CRC computation of sequences of bytes copied to ROM.
	 *
	 * @return the computed crc
	 */
	public static native int copyToROMCRCEnd();
}
