/*
 * Java
 *
 * Copyright 2011-2019 MicroEJ Corp. All rights reserved
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL Use is subject to license terms.
 */
package iceTea.lang.support;

/**
 * <p>Hook to handle specifically unrecoverable errors.</p>
 * <p>This default implementation just prints the error message {@link #INTERNAL_ERROR_MESSAGE}.</p>
 */
public class UnrecoverableErrorHandler {

	/**
	 * <p>System property used to specify the error handler class to use.</p>
	 */
	public static final String ERROR_HANDLER_CLASS_PROPERTY = "icetea.runtime.java.errorhandler";

	/**
	 * <p>Default error message print when an unrecoverable error occurs.</p>
	 */
	public static final String INTERNAL_ERROR_MESSAGE = "\"Internal limits reached. Please contact support@microej.com\"";

	private static Object ErrorMonitor = new Object();
	private static volatile int ErrorCount = 0;

	/**
	 * <p>Callback method called when an unexpected and unrecoverable error occurs.</p>
	 * <p>Subclass may override it but should call this default implementation which prints
	 * {@link #INTERNAL_ERROR_MESSAGE} on the standard error output if this is the first
	 * time we detect such error.</p>
	 * @param e the unrecoverable thrown exception
	 */
	public void handleError(Throwable e) {
		// displays the error message only if not yet done
		synchronized (ErrorMonitor) {
			if(++ErrorCount > 1) { // this is not the first time we handle an unrecoverable error
				// displays nothing
				return;
			}
			System.err.println(INTERNAL_ERROR_MESSAGE);
		}
	}

	/**
	 * <p>Instantiate the class denoted by the {@link #ERROR_HANDLER_CLASS_PROPERTY} property
	 * and use it as runtime handler.</p>
	 * <p>If an error occurs (property not set, class not visible, …), the default implementation
	 * is chosen.
	 * @return the error handler specified by the {@link #ERROR_HANDLER_CLASS_PROPERTY} property or the default
	 * 			implementation if an error occur.
	 */
	public static UnrecoverableErrorHandler getHandler() {
		String clazz = System.getProperty(ERROR_HANDLER_CLASS_PROPERTY);
		try {
			return (UnrecoverableErrorHandler) Class.forName(clazz).newInstance();
		} catch (Exception e) { // NullPointerException, InstantiationException, IllegalAccessException, ClassNotFoundException, …
			return new UnrecoverableErrorHandler(); // default implementation used as error handler if any problem occurs
		}
	}

}
