
package iceTea.lang;

import iceTea.lang.support.I2JBestFitAllocatorMapping;
import iceTea.lang.support.I2JCheckConstants;
import iceTea.lang.support.I2jConstants;
import iceTea.lang.support.IceteaRuntimeSupport;
import iceTea.lang.support.TraceHelper;

import com.ist.allocator.BestFitAllocator;

/**
 *
 * Java
 *
 * Copyright 2008 IS2T. All rights reserved
 * IS2T PROPRIETARY/ Use is subject to license terms.
 *
 * IceTea Object are generated as Java object
 * Struct and array of baseType are simulated directly in Memories
 *
 * Memories spec :
 *
 *
 *  One array of int split in ram, rom and heap area
 *  	-> ram is used for initial allocation at [0, SYMBOL_BEGIN_ROM[
 *  	-> rom is used for initial data allocation at [SYMBOL_BEGIN_ROM, SYMBOL_BEGIN_HEAP[
 *  	-> heap is used for dynamic allocation (new byte[100] for example) at [ SYMBOL_BEGIN_HEAP, SYMBOL_END_MEMORIES[
 *
 *  Type size :
 *  		sizeOf(byte) = 1
 *  		sizeOf(short) = 2
 *  		sizeOf( int) = 4
 *  		sizeOf(long) = 8
 *
 *  All array allocated  in Heap Memories should be int aligned

 *  To do a check when access to a value, a helperTable is created (mapped on Memories) : when a value is stored
 *  	in Memories[index], helperTable[index] store the type of the value (T_INT ...)
 *  	If helperTable[index] = 0 the Memories[index] doesn't fit a regular value...
 *
 *  Byte storage in Memories on 1 index (only low byte of Memories[index] is used )
 *  		setByte( index, byte) : Memories[index] == (int) byte
 *  		getByte( index)		 :  (byte) Memories[index];
 *
 *  Short/Char storage in Memories  on 2 indexes (only 2 low bytes of Memories[index] is used, Memories[index+1] is undefine):
 *  		setShort( index, short) : Memories[index] == (int) short
 *  		getByte( index)		 	:  (short) Memories[index];
 *  	->check index%2 == 0
 *
 *  Int storage in Memories on 4 indexes (only Memories[index] is used, Memories[index+1] to Memories[index+3] is undefine):
 *  		setInt( index, int) : Memories[index] == int
 *  		getInt( index)		:  Memories[index];
 *  	-> check index%4 == 0
 *
 *  Long storage in Memories on 8 indexes (only Memories[index] and Memories[index*1] are used, Memories[index+2] to Memories[index+7] is undefine):
 *  		setLong( index, long) : Memories[index] == (hight) long; Memories[index] == (low) long
 *  		getLong( index)		:  Memories[index] << 32 ||  Memories[index+1];
 *  	-> check index%4 == 0
 *
 *
 *  For example :
 *  	struct MyStruct
 *  		byte b1;
 *  		byte b2;
 *  		short s1;
 *  		int i1;
 *  		long l1;
 *  		an instance of Struct at startAddress =  4
 *
 *  	check : startAddress%4 = 0;
 *
 *  	Memories[4] = b1
 *  	Memories[5] = b2:
 *  	Memories[6] & Memories[7] = S2
 *  	Memories[8] ... Memories[11] = i1
 *  	Memories[12] ... Memories[20] = l1
 *
 *
 */
public class Memories implements I2jConstants {

	private static final boolean DEBUG_INCONSISTANCY = false;

	// s3 used an int oriented memories
	public static byte[] byteMemory;
	public static int[] intMemory;

	// debug inconsistency
	public static byte[] setTypes;
	public static final byte TYPE_NONE = 0;
	public static final byte TYPE_BOOLEAN = 1;
	public static final byte TYPE_BYTE = 2;
	public static final byte TYPE_CHAR = 3;
	public static final byte TYPE_SHORT = TYPE_CHAR;//4; // FIXME optimize: ok for the current memory model
	public static final byte TYPE_INT = 5;
	public static final byte TYPE_FLOAT = TYPE_INT;//6; // FIXME optimize: ok for the current memory model
	public static final byte TYPE_LONG = TYPE_INT;//7; // FIXME optimize: ok for the current memory model
	public static final byte TYPE_DOUBLE = TYPE_INT;//8; // FIXME optimize: ok for the current memory model

	static {
		initialize();
	}

	public static void initialize() {
		if (intMemory == null) {

			byteMemory = new byte[I2JBestFitAllocatorMapping.HeapEnd];
			intMemory = new int[I2JBestFitAllocatorMapping.HeapEnd >> 2];
			if (DEBUG_INCONSISTANCY) {
				setTypes = new byte[I2JBestFitAllocatorMapping.HeapEnd];
			}
			BestFitAllocator.initialize();
		}
	}

	public static boolean isDisposed() {
		return intMemory == null;
	}

	public static void dispose() {
		byteMemory = null;
		intMemory = null;
		if (DEBUG_INCONSISTANCY) {
			setTypes = null;
		}
	}

	/*---------------------
	 * Consistancy api
	 */

	private static void storeInMemory(int offset, byte type) {
		// if(offset == 6195148 && type == TYPE_BYTE) {
		// new java.lang.Exception().printStackTrace();
		// }
		setTypes[offset] = type;
	}

	private static void checkInMemory(int offset, byte type) {
		byte setType = setTypes[offset];
		if (setType != TYPE_NONE && setType != type) {
			java.lang.System.out.println("Read " + getType(type) + " instead of " + getType(setType) + " at " + offset);
			new java.lang.RuntimeException().printStackTrace();
			// throw new IllegalArgumentException("Bad memory access");
		}
	}

	private static java.lang.String getType(byte type) {
		switch (type) {
		case TYPE_BOOLEAN:
			return "BOOLEAN";
		case TYPE_BYTE:
			return "BYTE";
		case TYPE_CHAR:
			return "CHAR";
//		case TYPE_SHORT:
//			return "SHORT";
		case TYPE_INT:
			return "INT";
//		case TYPE_FLOAT:
//			return "FLOAT";
//		case TYPE_LONG:
//			return "LONG";
//		case TYPE_DOUBLE:
//			return "DOUBLE";
		default:
			return "unknown";
		}
	}

	/*---------------------
	 * Ram api
	 */
	public static void linkTo(int chunck, int chunckToLinkTo) {
		// just for evaluate arguments in case of side effect...
	}

	public static int getChunckAddress(int chunck, int address) {
		// just for evaluate arguments in case of side effect...
		return address;
	}

	/*---------------------
	 * Memories getter and setter
	 */

	public static int setInt(int offset, Object value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_INT);
		}
		// IceteaRuntimeSupport.precondCheck( value == null);
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, INT_SIZE);
		}
		intMemory[offset >> 2] = 0;
		return 0;
	}

	public static int getInt(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_INT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, INT_SIZE);
		}
		return intMemory[offset >> 2];
	}

	public static int setInt(int offset, int value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_INT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, INT_SIZE);
		}
		intMemory[offset >> 2] = value;
		return value;
	}

	public static int getInt(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_INT);
		}
		// used if ram1.setInt() is used with ram1 an expression to evaluate
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, INT_SIZE);
		}
		return intMemory[offset >> 2];
	}

	public static int setInt(int ramReceiver, int offset, int value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_INT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, INT_SIZE);
		}
		intMemory[offset >> 2] = value;
		return value;
	}

	public static byte getByte(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_BYTE);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, BYTE_SIZE);
		}
		return byteMemory[offset];
	}

	public static byte setByte(int offset, byte value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_BYTE);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, BYTE_SIZE);
		}
		byteMemory[offset] = value;
		return value;
	}

	public static byte getByte(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_BYTE);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, BYTE_SIZE);
		}
		return byteMemory[offset];
	}

	public static byte setByte(int ramReceiver, int offset, byte value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_BYTE);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, BYTE_SIZE);
		}
		byteMemory[offset] = value;
		return value;
	}

	public static boolean getBoolean(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_BOOLEAN);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, BOOLEAN_SIZE);
		}
		return byteMemory[offset] == TRUE ? true : false;
	}

	public static boolean setBoolean(int offset, boolean value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_BOOLEAN);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, BOOLEAN_SIZE);
		}
		byteMemory[offset] = value ? TRUE : FALSE;
		return value;
	}

	public static boolean getBoolean(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_BOOLEAN);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, BOOLEAN_SIZE);
		}
		return byteMemory[offset] == TRUE ? true : false;
	}

	public static boolean setBoolean(int ramReceiver, int offset, boolean value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_BOOLEAN);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, BOOLEAN_SIZE);
		}
		byteMemory[offset] = value ? TRUE : FALSE;
		return value;
	}

	public static char getChar(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_CHAR);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, CHAR_SIZE);
		}
		return (char) (((int) byteMemory[offset] << 8) | ((int) byteMemory[offset + 1] & 0xff));
	}

	public static char setChar(int offset, char value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_CHAR);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, CHAR_SIZE);
		}
		byteMemory[offset] = (byte) (value >> 8);
		byteMemory[offset + 1] = (byte) value;
		return value;
	}

	public static char getChar(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_CHAR);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, CHAR_SIZE);
		}
		return (char) (((int) byteMemory[offset] << 8) | ((int) byteMemory[offset + 1] & 0xff));
	}

	public static char setChar(int ramReceiver, int offset, char value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_CHAR);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, CHAR_SIZE);
		}
		byteMemory[offset] = (byte) (value >> 8);
		byteMemory[offset + 1] = (byte) value;
		return value;
	}

	public static short getShort(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_SHORT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, SHORT_SIZE);
		}
		return (short) (((int) byteMemory[offset] << 8) | ((int) byteMemory[offset + 1] & 0xff));
	}

	public static short setShort(int offset, short value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_SHORT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, SHORT_SIZE);
		}
		byteMemory[offset] = (byte) (value >> 8);
		byteMemory[offset + 1] = (byte) value;
		return value;
	}

	public static short getShort(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_SHORT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, SHORT_SIZE);
		}
		return (short) (((int) byteMemory[offset] << 8) | ((int) byteMemory[offset + 1] & 0xff));
	}

	public static short setShort(int ramReceiver, int offset, short value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_SHORT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, SHORT_SIZE);
		}
		byteMemory[offset] = (byte) (value >> 8);
		byteMemory[offset + 1] = (byte) value;
		return value;
	}

	public static float getFloat(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_FLOAT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, FLOAT_SIZE);
		}
		return Float.intBitsToFloat(intMemory[offset >> 2]);
	}

	public static float setFloat(int offset, float value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_FLOAT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, FLOAT_SIZE);
		}
		intMemory[offset >> 2] = Float.floatToIntBits(value);
		return value;
	}

	public static float getFloat(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_FLOAT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, FLOAT_SIZE);
		}
		return Float.intBitsToFloat(intMemory[offset >> 2]);
	}

	public static float setFloat(int ramReceiver, int offset, float value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_FLOAT);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, FLOAT_SIZE);
		}
		intMemory[offset >> 2] = Float.floatToIntBits(value);
		return value;
	}

	public static long getLong(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_LONG);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, LONG_SIZE);
		}
		return ((long) intMemory[offset >>= 2] << 32) | ((long) intMemory[offset + 1] & 0xffffffffl);
	}

	public static long setLong(int offset, long value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_LONG);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, LONG_SIZE);
		}
		intMemory[offset >>= 2] = (int) (value >> 32);
		intMemory[offset + 1] = (int) value;
		return value;
	};

	public static long getLong(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_LONG);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 2)
				TraceHelper.checkAccessTo(offset, LONG_SIZE);
		}
		return ((long) intMemory[offset >>= 2] << 32) | ((long) intMemory[offset + 1] & 0xffffffffl);
	}

	public static long setLong(int ramReceiver, int offset, long value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_LONG);
		}
		if (I2JCheckConstants.ACTIVATE_CHECK) {
			if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
				TraceHelper.checkAccessTo(offset, LONG_SIZE);
		}
		intMemory[offset >>= 2] = (int) (value >> 32);
		intMemory[offset + 1] = (int) value;
		return value;
	}

	public static double getDouble(int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_DOUBLE);
		}
		return Double.longBitsToDouble(((long) intMemory[offset >>= 2] << 32) | ((long) intMemory[offset + 1] & 0xffffffffl));
	}

	public static double setDouble(int offset, double value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_DOUBLE);
		}
		long v = Double.doubleToLongBits(value);
		intMemory[offset >>= 2] = (int) (v >> 32);
		intMemory[offset + 1] = (int) v;
		return value;
	}

	public static double getDouble(int ramReceiver, int offset) {
		if (DEBUG_INCONSISTANCY) {
			checkInMemory(offset, TYPE_DOUBLE);
		}
		return Double.longBitsToDouble(((long) intMemory[offset >>= 2] << 32) | ((long) intMemory[offset + 1] & 0xffffffffl));
	}

	public static double setDouble(int ramReceiver, int offset, double value) {
		if (DEBUG_INCONSISTANCY) {
			storeInMemory(offset, TYPE_DOUBLE);
		}
		long v = Double.doubleToLongBits(value);
		intMemory[offset >>= 2] = (int) (v >> 32);
		intMemory[offset + 1] = (int) v;
		return value;
	}

	/*---------------------
	 * Memories pre and post incrementation
	 */
	public static long setPostIncrLong(int offset) {
		long value = getLong(offset);
		setLong(offset, value + 1);
		return value;
	}

	public static int setPostIncrInt(int offset) {
		int value = getInt(offset);
		setInt(offset, value + 1);
		return value;
	}

	public static char setPostIncrChar(int offset) {
		char value = getChar(offset);
		setChar(offset, (char) (value + 1));
		return value;
	}

	public static short setPostIncrShort(int offset) {
		short value = getShort(offset);
		setShort(offset, (short) (value + 1));
		return value;
	}

	public static byte setPostIncrByte(int offset) {
		byte value = getByte(offset);
		setByte(offset, (byte) (value + 1));
		return value;
	}

	public static long setPreIncrLong(int offset) {
		long value = getLong(offset);
		setLong(offset, ++value);
		return value;
	}

	public static int setPreIncrInt(int offset) {
		int value = getInt(offset);
		setInt(offset, ++value);
		return value;
	}

	public static char setPreIncrChar(int offset) {
		char value = getChar(offset);
		setChar(offset, (char) (++value));
		return value;
	}

	public static short setPreIncrShort(int offset) {
		short value = getShort(offset);
		setShort(offset, (short) (++value));
		return value;
	}

	public static byte setPreIncrByte(int offset) {
		byte value = getByte(offset);
		setByte(offset, (byte) (++value));
		return value;
	}

	public static long setPreDecrLong(int offset) {
		long value = getLong(offset);
		setLong(offset, --value);
		return value;
	}

	public static int setPreDecrInt(int offset) {
		int value = getInt(offset);
		setInt(offset, --value);
		return value;
	}

	public static char setPreDecrChar(int offset) {
		char value = getChar(offset);
		setChar(offset, (char) (--value));
		return value;
	}

	public static short setPreDecrShort(int offset) {
		short value = getShort(offset);
		setShort(offset, (short) (--value));
		return value;
	}

	public static byte setPreDecrByte(int offset) {
		byte value = getByte(offset);
		setByte(offset, (byte) (--value));
		return value;
	}

	public static long setPostDecrLong(int offset) {
		long value = getLong(offset);
		setLong(offset, value - 1);
		return value;
	}

	public static int setPostDecrInt(int offset) {
		int value = getInt(offset);
		setInt(offset, value - 1);
		return value;
	}

	public static char setPostDecrChar(int offset) {
		char value = getChar(offset);
		setChar(offset, (char) (value - 1));
		return value;
	}

	public static short setPostDecrShort(int offset) {
		short value = getShort(offset);
		setShort(offset, (short) (value - 1));
		return value;
	}

	public static byte setPostDecrByte(int offset) {
		byte value = getByte(offset);
		setByte(offset, (byte) (value - 1));
		return value;
	}

	/*
	 * --------------------------------- CompoundAssignment
	 */
	public static long setBitwiseAndLong(int offset, long value) {
		value &= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setBitwiseAndInt(int offset, int value) {
		value &= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static boolean setBitwiseAndBoolean(int offset, boolean value) {
		value &= getBoolean(offset);
		setBoolean(offset, value);
		return value;
	}

	public static char setBitwiseAndChar(int offset, char value) {
		value &= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setBitwiseAndShort(int offset, short value) {
		value &= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setBitwiseAndByte(int offset, byte value) {
		value &= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static long setBitwiseXorLong(int offset, long value) {
		value ^= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setBitwiseXorInt(int offset, int value) {
		value ^= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static boolean setBitwiseXorBoolean(int offset, boolean value) {
		value ^= getBoolean(offset);
		setBoolean(offset, value);
		return value;
	}

	public static char setBitwiseXorChar(int offset, char value) {
		value ^= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setBitwiseXorShort(int offset, short value) {
		value ^= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setBitwiseXorByte(int offset, byte value) {
		value ^= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static long setBitwiseOrLong(int offset, long value) {
		value |= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setBitwiseOrInt(int offset, int value) {
		value |= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static boolean setBitwiseOrBoolean(int offset, boolean value) {
		value |= getBoolean(offset);
		setBoolean(offset, value);
		return value;
	}

	public static char setBitwiseOrChar(int offset, char value) {
		value |= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setBitwiseOrShort(int offset, short value) {
		value |= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setBitwiseOrByte(int offset, byte value) {
		value |= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static double setRemainderDouble(int offset, double value) {
		value %= getDouble(offset);
		setDouble(offset, value);
		return value;
	}

	public static long setRemainderLong(int offset, long value) {
		value %= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setRemainderInt(int offset, int value) {
		value %= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static float setRemainderFloat(int offset, float value) {
		value %= getFloat(offset);
		setFloat(offset, value);
		return value;
	}

	public static char setRemainderChar(int offset, char value) {
		value %= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setRemainderShort(int offset, short value) {
		value %= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setRemainderByte(int offset, byte value) {
		value %= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static double setPlusDouble(int offset, double value) {
		value += getDouble(offset);
		setDouble(offset, value);
		return value;
	}

	public static long setPlusLong(int offset, long value) {
		value += getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setPlusInt(int offset, int value) {
		value += getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static float setPlusFloat(int offset, float value) {
		value += getFloat(offset);
		setFloat(offset, value);
		return value;
	}

	public static short setPlusShort(int offset, short value) {
		value += getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setPlusByte(int offset, byte value) {
		value += getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static char setPlusChar(int offset, char value) {
		value += getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static double setMinusDouble(int offset, double value) {
		value -= getDouble(offset);
		setDouble(offset, value);
		return value;
	}

	public static long setMinusLong(int offset, long value) {
		value -= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setMinusInt(int offset, int value) {
		value -= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static float setMinusFloat(int offset, float value) {
		value -= getFloat(offset);
		setFloat(offset, value);
		return value;
	}

	public static char setMinusChar(int offset, char value) {
		value -= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setMinusShort(int offset, short value) {
		value -= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setMinusByte(int offset, byte value) {
		value -= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static double setMultiplyDouble(int offset, double value) {
		value *= getDouble(offset);
		setDouble(offset, value);
		return value;
	}

	public static long setMultiplyLong(int offset, long value) {
		value *= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setMultiplyInt(int offset, int value) {
		value *= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static float setMultiplyFloat(int offset, float value) {
		value *= getFloat(offset);
		setFloat(offset, value);
		return value;
	}

	public static char setMultiplyChar(int offset, char value) {
		value *= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setMultiplyShort(int offset, short value) {
		value *= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setMultiplyByte(int offset, byte value) {
		value *= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static double setDivideDouble(int offset, double value) {
		value /= getDouble(offset);
		setDouble(offset, value);
		return value;
	}

	public static long setDivideLong(int offset, long value) {
		value /= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setDivideInt(int offset, int value) {
		value /= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static float setDivideFloat(int offset, float value) {
		value /= getFloat(offset);
		setFloat(offset, value);
		return value;
	}

	public static char setDivideChar(int offset, char value) {
		value /= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setDivideShort(int offset, short value) {
		value /= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setDivideByte(int offset, byte value) {
		value /= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static long setLeftShiftLong(int offset, long value) {
		value <<= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setLeftShiftInt(int offset, int value) {
		value <<= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static char setLeftShiftChar(int offset, char value) {
		value <<= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setLeftShiftShort(int offset, short value) {
		value <<= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setLeftShiftByte(int offset, byte value) {
		value <<= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static long setRightShiftLong(int offset, long value) {
		value >>= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setRightShiftInt(int offset, int value) {
		value >>= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static char setRightShiftChar(int offset, char value) {
		value >>= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setRightShiftShort(int offset, short value) {
		value >>= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setRightShiftByte(int offset, byte value) {
		value >>= getByte(offset);
		setByte(offset, value);
		return value;
	}

	public static long setUnsignedRightShiftLong(int offset, long value) {
		value >>>= getLong(offset);
		setLong(offset, value);
		return value;
	}

	public static int setUnsignedRightShiftInt(int offset, int value) {
		value >>>= getInt(offset);
		setInt(offset, value);
		return value;
	}

	public static char setUnsignedRightShiftChar(int offset, char value) {
		value >>>= getChar(offset);
		setChar(offset, value);
		return value;
	}

	public static short setUnsignedRightShiftShort(int offset, short value) {
		value >>>= getShort(offset);
		setShort(offset, value);
		return value;
	}

	public static byte setUnsignedRightShiftByte(int offset, byte value) {
		value >>>= getByte(offset);
		setByte(offset, value);
		return value;
	}

	/*
	 * ----------------------------- IceTea API
	 */
	public static int i2j_startAddress(int start, int stop, int ram) {
		return start;
	}

	public static int ram() {
		return I2JBestFitAllocatorMapping.HeapStart;
	}

	public static int rom() {
		return I2JBestFitAllocatorMapping.HeapStart;
	}

	/**
	 * Copy nbByte from ptrSrc to ptrDesr
	 *
	 * @param src
	 *            : unused but permits to evaluate the first argument - if side effect -
	 * @param ptrSrc
	 *            : first address to copy
	 * @param dest
	 *            : unused but permits to evaluate the first argument - if side effect -
	 * @param ptrDest
	 *            : first address of destination
	 * @param nbByte
	 *            : nbByte to copy
	 */
	public static void memcpy(int src, int ptrSrc, int dest, int ptrDest, int nbByte) {
		java.lang.System.arraycopy(byteMemory, ptrSrc, byteMemory, ptrDest, nbByte);
		java.lang.System.arraycopy(intMemory, ptrSrc >> 2, intMemory, ptrDest >> 2, nbByte >> 2); // FIXME validate division
		if(DEBUG_INCONSISTANCY) {
			System.arraycopy(setTypes, ptrSrc, setTypes, ptrDest, nbByte);
		}
	}

	public static void memclear(int src, int ptrSrc, int nbByte) {
		while (--nbByte >= 0) {
			if (I2JCheckConstants.ACTIVATE_CHECK) {
				if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
					TraceHelper.checkAccessTo(ptrSrc, BYTE_SIZE);
			}
			if(DEBUG_INCONSISTANCY) {
				storeInMemory(ptrSrc, TYPE_NONE);
			}
			if ((ptrSrc & 3) == 0) {
				intMemory[ptrSrc >> 2] = 0;
			}
			byteMemory[ptrSrc++] = 0;
		}
	}

	public static void memset(int src, int ptrSrc, int nbByte, byte value) {
		while (--nbByte >= 0) {
			if (I2JCheckConstants.ACTIVATE_CHECK) {
				if (TraceHelper.BREAKPOINT_ON_MEMORIES == 1)
					TraceHelper.checkAccessTo(ptrSrc, BYTE_SIZE);
			}
			if(DEBUG_INCONSISTANCY) {
				// assume that memset is byte oriented
				storeInMemory(ptrSrc, TYPE_BYTE);
			}
			byteMemory[ptrSrc++] = value;
		}
	}

	/**
	 * Check bounds before array access like array[index]
	 *
	 * @param header
	 *            : array address
	 * @param index
	 *            : index in array
	 * @param elementSize
	 *            : size in byte of array elements
	 * @return absolute address of array[index]
	 */
	public static int checkBounds(int header, int index, int elementSize) {
		if (index < 0) {
			// not in icetea specification so thrown a java exception
			throw new ArrayIndexOutOfBoundsException("@array: " + header + " index: " + index);
		}

		int length = intMemory[(header - INT_SIZE) >> 2];
		if (index >= length) {
			// not in icetea specification so thrown a java exception
			throw new ArrayIndexOutOfBoundsException("@array: " + header + " length: " + length + " index: " + index);
		}

		return header + (index * elementSize);
	}

	public static int checkNullPointers(int header) {
		if (header == 0) {
			// not in icetea specification so thrown a java exception
			throw new NullPointerException();
		}
		return header;
	}

	/*
	 * ----------------------------- For this kind of code the CompoundAssignment should have 3 parameters int[] tabInt = new int[] {9,9,9}; tabInt[0] += ( tabInt[0] = 3);
	 *
	 * -> at end tabInt[0] = 12 tabInt[0] should be saved before execute tabInt[0] = 3
	 */
	public static double setPlusDouble(int offset, double saveValue, double addValue) {
		saveValue += addValue;
		setDouble(offset, saveValue);
		return saveValue;
	}

	public static long setPlusLong(int offset, long saveValue, long addValue) {
		saveValue += addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setPlusInt(int offset, int saveValue, int addValue) {
		saveValue += addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static float setPlusFloat(int offset, float saveValue, float addValue) {
		saveValue += addValue;
		setFloat(offset, saveValue);
		return saveValue;
	}

	public static char setPlusChar(int offset, char saveValue, char addValue) {
		saveValue += addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setPlusShort(int offset, short saveValue, short addValue) {
		saveValue += addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setPlusByte(int offset, byte saveValue, byte addValue) {
		saveValue += addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static double setMinusDouble(int offset, double saveValue, double addValue) {
		saveValue -= addValue;
		setDouble(offset, saveValue);
		return saveValue;
	}

	public static long setMinusLong(int offset, long saveValue, long addValue) {
		saveValue -= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setMinusInt(int offset, int saveValue, int addValue) {
		saveValue -= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static float setMinusFloat(int offset, float saveValue, float addValue) {
		saveValue -= addValue;
		setFloat(offset, saveValue);
		return saveValue;
	}

	public static char setMinusChar(int offset, char saveValue, char addValue) {
		saveValue -= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setMinusShort(int offset, short saveValue, short addValue) {
		saveValue -= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setMinusByte(int offset, byte saveValue, byte addValue) {
		saveValue -= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static double setMultiplyDouble(int offset, double saveValue, double addValue) {
		saveValue *= addValue;
		setDouble(offset, saveValue);
		return saveValue;
	}

	public static long setMultiplyLong(int offset, long saveValue, long addValue) {
		saveValue *= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setMultiplyInt(int offset, int saveValue, int addValue) {
		saveValue *= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static float setMultiplyFloat(int offset, float saveValue, float addValue) {
		saveValue *= addValue;
		setFloat(offset, saveValue);
		return saveValue;
	}

	public static char setMultiplyChar(int offset, char saveValue, char addValue) {
		saveValue *= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setMultiplyShort(int offset, short saveValue, short addValue) {
		saveValue *= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setMultiplyByte(int offset, byte saveValue, byte addValue) {
		saveValue *= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static double setDivideDouble(int offset, double saveValue, double addValue) {
		saveValue /= addValue;
		setDouble(offset, saveValue);
		return saveValue;
	}

	public static long setDivideLong(int offset, long saveValue, long addValue) {
		saveValue /= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setDivideInt(int offset, int saveValue, int addValue) {
		saveValue /= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static float setDivideFloat(int offset, float saveValue, float addValue) {
		saveValue /= addValue;
		setFloat(offset, saveValue);
		return saveValue;
	}

	public static char setDivideChar(int offset, char saveValue, char addValue) {
		saveValue /= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setDivideShort(int offset, short saveValue, short addValue) {
		saveValue /= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setDivideByte(int offset, byte saveValue, byte addValue) {
		saveValue /= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static double setRemainderDouble(int offset, double saveValue, double addValue) {
		saveValue %= addValue;
		setDouble(offset, saveValue);
		return saveValue;
	}

	public static long setRemainderLong(int offset, long saveValue, long addValue) {
		saveValue %= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setRemainderInt(int offset, int saveValue, int addValue) {
		saveValue %= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static float setRemainderFloat(int offset, float saveValue, float addValue) {
		saveValue %= addValue;
		setFloat(offset, saveValue);
		return saveValue;
	}

	public static char setRemainderChar(int offset, char saveValue, char addValue) {
		saveValue %= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setRemainderShort(int offset, short saveValue, short addValue) {
		saveValue %= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setRemainderByte(int offset, byte saveValue, byte addValue) {
		saveValue %= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setBitwiseAndLong(int offset, long saveValue, long addValue) {
		saveValue &= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setBitwiseAndInt(int offset, int saveValue, int addValue) {
		saveValue &= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static boolean setBitwiseAndBoolean(int offset, boolean saveValue, boolean addValue) {
		saveValue &= addValue;
		setBoolean(offset, saveValue);
		return saveValue;
	}

	public static char setBitwiseAndChar(int offset, char saveValue, char addValue) {
		saveValue &= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setBitwiseAndShort(int offset, short saveValue, short addValue) {
		saveValue &= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setBitwiseAndByte(int offset, byte saveValue, byte addValue) {
		saveValue &= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setBitwiseOrLong(int offset, long saveValue, long addValue) {
		saveValue |= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setBitwiseOrInt(int offset, int saveValue, int addValue) {
		saveValue |= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static boolean setBitwiseOrBoolean(int offset, boolean saveValue, boolean addValue) {
		saveValue |= addValue;
		setBoolean(offset, saveValue);
		return saveValue;
	}

	public static char setBitwiseOrChar(int offset, char saveValue, char addValue) {
		saveValue |= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setBitwiseOrShort(int offset, short saveValue, short addValue) {
		saveValue |= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setBitwiseOrByte(int offset, byte saveValue, byte addValue) {
		saveValue |= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setBitwiseXorLong(int offset, long saveValue, long addValue) {
		saveValue ^= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setBitwiseXorInt(int offset, int saveValue, int addValue) {
		saveValue ^= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static boolean setBitwiseXorBoolean(int offset, boolean saveValue, boolean addValue) {
		saveValue ^= addValue;
		setBoolean(offset, saveValue);
		return saveValue;
	}

	public static char setBitwiseXorChar(int offset, char saveValue, char addValue) {
		saveValue ^= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setBitwiseXorShort(int offset, short saveValue, short addValue) {
		saveValue ^= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setBitwiseXorByte(int offset, byte saveValue, byte addValue) {
		saveValue ^= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setLeftShiftLong(int offset, long saveValue, long addValue) {
		saveValue <<= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setLeftShiftInt(int offset, int saveValue, int addValue) {
		saveValue <<= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static char setLeftShiftChar(int offset, char saveValue, char addValue) {
		saveValue <<= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setLeftShiftShort(int offset, short saveValue, short addValue) {
		saveValue <<= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setLeftShiftByte(int offset, byte saveValue, byte addValue) {
		saveValue <<= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setRightShiftLong(int offset, long saveValue, long addValue) {
		saveValue >>= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setRightShiftInt(int offset, int saveValue, int addValue) {
		saveValue >>= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static char setRightShiftChar(int offset, char saveValue, char addValue) {
		saveValue >>= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setRightShiftShort(int offset, short saveValue, short addValue) {
		saveValue >>= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setRightShiftByte(int offset, byte saveValue, byte addValue) {
		saveValue >>= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	public static long setUnsignedRightShiftLong(int offset, long saveValue, long addValue) {
		saveValue >>>= addValue;
		setLong(offset, saveValue);
		return saveValue;
	}

	public static int setUnsignedRightShiftInt(int offset, int saveValue, int addValue) {
		saveValue >>>= addValue;
		setInt(offset, saveValue);
		return saveValue;
	}

	public static char setUnsignedRightShiftChar(int offset, char saveValue, char addValue) {
		saveValue >>>= addValue;
		setChar(offset, saveValue);
		return saveValue;
	}

	public static short setUnsignedRightShiftShort(int offset, short saveValue, short addValue) {
		saveValue >>>= addValue;
		setShort(offset, saveValue);
		return saveValue;
	}

	public static byte setUnsignedRightShiftByte(int offset, byte saveValue, byte addValue) {
		saveValue >>>= addValue;
		setByte(offset, saveValue);
		return saveValue;
	}

	/**
	 * return all bytes of the array of byte at address offset in Memories
	 *
	 * @param offset
	 * @return new array of byte
	 */
	public static byte[] getBytes(int address) {
		int length = getInt(address - INT_SIZE);
		byte[] bytes = new byte[length];
		for (int i = 0; i < length; i++) {
			bytes[i] = getByte(address);
			address += BYTE_SIZE;
		}

		return bytes;
	}

	/*
	 * ----------------------------- debug methods for see array values choose the right method depending of element type !
	 */
	public static byte[] toJavaByte(int arrayHeader) {
		IceteaRuntimeSupport.precondCheck(arrayHeader != NULL);
		int size = Memories.getInt(arrayHeader - INT_SIZE);
		if (size < 0)
			return null;
		byte[] values = new byte[size];
		int ptr = arrayHeader + size * BYTE_SIZE;
		while (--size >= 0) {
			ptr -= BYTE_SIZE;
			values[size] = Memories.getByte(ptr);
		}
		return values;
	}

	public static short[] toJavaShort(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		short[] values = new short[size];
		int ptr = header + size * SHORT_SIZE;
		while (--size >= 0) {
			ptr -= SHORT_SIZE;
			values[size] = Memories.getShort(ptr);
		}
		return values;
	}

	public static char[] toJavaChar(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		char[] values = new char[size];
		int ptr = header + size * CHAR_SIZE;
		while (--size >= 0) {
			ptr -= CHAR_SIZE;
			values[size] = Memories.getChar(ptr);
		}
		return values;
	}

	public static boolean[] toJavaBoolean(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		boolean[] values = new boolean[size];
		int ptr = header + size * BOOLEAN_SIZE;
		while (--size >= 0) {
			ptr -= BOOLEAN_SIZE;
			values[size] = Memories.getBoolean(ptr);
		}
		return values;
	}

	public static int[] toJavaInt(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		int[] values = new int[size];
		int ptr = header + size * INT_SIZE;
		while (--size >= 0) {
			ptr -= INT_SIZE;
			values[size] = Memories.getInt(ptr);
		}
		return values;
	}

	public static float[] toJavaFloat(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		float[] values = new float[size];
		int ptr = header + size * INT_SIZE;
		while (--size >= 0) {
			ptr -= INT_SIZE;
			values[size] = Memories.getFloat(ptr);
		}
		return values;
	}

	public static long[] toJavaLong(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		long[] values = new long[size];
		int ptr = header + size * LONG_SIZE;
		while (--size >= 0) {
			ptr -= LONG_SIZE;
			values[size] = Memories.getLong(ptr);
		}
		return values;
	}

	public static double[] toJavaDouble(int header) {
		IceteaRuntimeSupport.precondCheck(header != NULL);
		int size = Memories.getInt(header - INT_SIZE);
		if (size < 0)
			return null;
		double[] values = new double[size];
		int ptr = header + size * DOUBLE_SIZE;
		while (--size >= 0) {
			ptr -= DOUBLE_SIZE;
			values[size] = Memories.getDouble(ptr);
		}
		return values;
	}

	public static void printMemories(int start, int stop) {
		java.lang.StringBuilder sb = new java.lang.StringBuilder();
		for (int i = start; i < stop; i++) {
			if (i % 100 == 0)
				sb.append("\n");
			else if (i % 4 == 0)
				sb.append(' ');
			sb.append(Memories.getInt(i));
			sb.append(',');
		}
		java.lang.System.out.println(sb.toString());
	}

}