/*
 * Created on 24-Mar-2004
 */
package iceTea.io;
import iceTea.lang.String;
import iceTea.lang.support.IceteaStringSupport;
/**
 * @author IST
 * 
 * A PrintStream adds functionality to another output stream, namely the
 * ability to print representations of various data values conveniently. Two
 * other features are provided as well. Unlike other output streams, a
 * PrintStream never throws an IOException; instead, exceptional situations
 * merely set an internal flag that can be tested via the checkError method.
 * 
 * All characters printed by a PrintStream are converted into bytes using the
 * platform's default character encoding.
 */

public class PrintStream extends OutputStream {

	/* CLASS CONSTANTS */
	
	/* Every call to println() need that constant */
	private static final byte lineSep = '\n'; //System.getProperty("line.separator").chars ;
	
	
	/* INSTANCE VARIABLES */
	
	/* The Writer for printing chars */
	private OutputStreamWriter outWriter ;
	
	/* The error marker which is true when an error has occured */
	private boolean errorFlag ;
	
	
	
    /**
     * Create a new print stream. This stream will not flush automatically.
     */
    public PrintStream(OutputStream out) {
    	this.outWriter = new OutputStreamWriter(out) ;
    }

    /**
     * Flush the stream and check its error state. The internal error state is
     * set to true when the underlying output stream throws an IOException, and
     * when the setError method is invoked.
     */
    public boolean checkError() {
    	try {
    		synchronized(this) {
    			//if (this.outWriter != null){
    				outWriter.flush();
    			//}
    		}
    		return this.errorFlag ;
    	}
    	catch(IOException e) {
    		return this.errorFlag=true ;
    	}
    }

    /**
     * Close the stream. This is done by flushing the stream and then closing
     * the underlying output stream.
     */
    public void close() {
    	try {
    		//if (this.outWriter != null) {
    		    this.outWriter.flush() ; 
    			this.outWriter.close() ;
    		//}
    	}
    	catch (IOException e) {
    		this.errorFlag = true ;
    	}
    }

    /**
     * Flush the stream. This is done by writing any buffered output bytes to
     * the underlying output stream and then flushing that stream.
     */
    public void flush() {
    	try {
    		//if (this.outWriter != null)
    			this.outWriter.flush() ;
    	}
    	catch (IOException e) {
    		this.errorFlag = true ;
    	}
    }
    /**
     * Print a double-precision floating point number. The string produced by
     * String.valueOf(double) is translated into bytes according to the
     * platform's default character encoding, and these bytes are written in
     * exactly the manner of the write(int) method.
     */
    public void print(double d) {
    	print(Double.toString(d)) ;
    }

    /**
     * Print a floating point number. The string produced by
     * String.valueOf(float) is translated into bytes according to the
     * platform's default character encoding, and these bytes are written in
     * exactly the manner of the write(int) method.
     */
    public void print(float f) {
    	print(Float.toString(f)) ;
    }



    

    /**
     * Print an object. The string produced by the String.valueOf(Object)
     * method is translated into bytes according to the platform's default
     * character encoding, and these bytes are written in exactly the manner of
     * the write(int) method.
     */
    public void print(Object obj) {
    	print(obj.toString()) ;
    }

    

   
 



 
    
    /**
     * Print a double and then terminate the line. This method behaves as
     * though it invokes print(double) and then println().
     */
    public void println(double x) {
    	print(x) ;
    	println() ;
    }

    /**
     * Print a float and then terminate the line. This method behaves as though
     * it invokes print(float) and then println().
     */
    public void println(float x) {
    	print(x) ;
    	println() ;
    }

 

 

    /**
     * Print an Object and then terminate the line. This method behaves as
     * though it invokes print(Object) and then println().
     */
    public void println(Object x) {
    	print(x) ;
    	println() ;
    }



    /**
     * Set the error state of the stream to true.
     */
    protected void setError() {
    	this.errorFlag = true ;
    }

    
    /**
     * Write the specified byte to this stream.
     * 
     * Note that the byte is written as given; to write a character that will
     * be translated according to the platform's default character encoding,
     * use the print(char) or println(char) methods.
     */
	public void write(int b) {
		try {
			// Use outStream to print the byte as given
			this.outWriter.out.write(b);
		} catch (IOException e) {
			this.errorFlag = true;
		}
	}
	
    /**
     * Print a boolean value. The string produced by String.valueOf(boolean) is
     * translated into bytes according to the platform's default character
     * encoding, and these bytes are written in exactly the manner of the
     * write(int) method.
     */
    public void print(boolean b) {
    	if (b)
    		print("true") ;
    	else
    		print("false") ;
    }
	public void print(byte b){
		print((int)b);
	}
	
	public void print(short b){
		print((int)b);
	}
	
    public void printAsciiCharacter(byte b){
        print((char)b);
    }
    /**
     * Print a character. The character is translated into one or more bytes
     * according to the platform's default character encoding, and these bytes
     * are written in exactly the manner of the write(int) method.
     */
    public void print(char c) {
    	print(new char[]{c}) ;
    }
	public void printCharValue(char c){
		// print the unsigned value instead of the character that it represents
		print((int)c);
	}
    /**
     * Print an integer. The string produced by String.valueOf(int) is
     * translated into bytes according to the platform's default character
     * encoding, and these bytes are written in exactly the manner of the
     * write(int) method.
     */
    public void print(int i) {
    	print(Integer.toString(i)) ;
    }
    /**
     * Print a long integer. The string produced by String.valueOf(long) is
     * translated into bytes according to the platform's default character
     * encoding, and these bytes are written in exactly the manner of the
     * write(int) method.
     */
    public void print(long l) {
    	print(Long.toString(l)) ;
    }
    /**
     * Print a string. If the argument is null then the string "null" is
     * printed. Otherwise, the string's characters are converted into bytes
     * according to the platform's default character encoding, and these bytes
     * are written in exactly the manner of the write(int) method.
     */
    public void print(String s) {
    	if (s==null)
    		print("null") ;
    	else{
    		try{
    		//TODO To optimize
    			byte[] bytes = IceteaStringSupport.toJavaBytes( s);
    			this.outWriter.write( bytes, 0, s.length);
    		}
    		catch(IOException e)
			{
    			errorFlag = true;
    		}
    	}
    }
    public void printFree(String s){
    	print(s);
    	s.dispose();
    }
  
	public void print(byte[] s){
		print(s, (short)0, (short)s.length);		
	}
	public void print(byte[] s, short offset, short length){
		print(s, (int)offset, (int)length);
	}
	public void print(byte[] s, int offset, int length){
		try{ outWriter.write(s, offset, length); }
		catch(IOException e){ e.printStackTrace(); }
	}

		
	/**
	 * Terminate the current line by writing the line separator string. The
	 * line separator string is defined by the system property line.separator,
	 * and is not necessarily a single newline character ('\n').
	 */
	public void println() {
		print(lineSep) ;
    }
	/**
	 * Print a boolean and then terminate the line. This method behaves as
	 * though it invokes print(boolean) and then println().
	 */
	public void println(boolean x) {
		print(x) ;
		println() ;
    }
    public void println(byte b){
    	print(b);
    	println();
    }
    
	public void println(short b){
		print((int)b);
	}
    
    
    public void printlnAsciiCharacter(byte b){
        printAsciiCharacter(b);
        println();
    }
    /**
     * Print a character and then terminate the line. This method behaves as
     * though it invokes print(char) and then println().
     */
    public void println(char x) {
    	print(x) ;
    	println() ;
    }
    public void printlnCharValue(char c){
    	printCharValue(c);
    	println();
    }
    /**
     * Print an integer and then terminate the line. This method behaves as
     * though it invokes print(int) and then println().
     */
    public void println(int x) {
    	print(x) ;
    	println() ;
    }
    /**
     * Print a long and then terminate the line. This method behaves as though
     * it invokes print(long) and then println().
     */
    public void println(long x) {
    	print(x) ;
    	println() ;
    }
    /**
     * Print a String and then terminate the line. This method behaves as
     * though it invokes print(String) and then println().
     */
    public void println(String x) {
    	print(x) ;
    	println() ;
    }
    public void printlnFree(String s){
    	printFree(s);
    	println();
    }
    /**
     * Print an array character and then terminate the line. This method behaves as
     * though it invokes print(char[]) and then println().
     */
    public void println(char[] x) {
    	print(x) ;
    	println() ;
    }    
    public void println(byte[] s){
    	print(s);
    	println();
    }
	public void println(byte[] chars, short offset, short length){
		print(chars, offset, length);
    	println();
	}
	public void println(byte[] chars, int offset, int length){
		println(chars, (int)offset, (int)length);
	}
}
