
/*
 *	Java
 *
 *	Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 *	This library is provided in source code for use, modification and test, subject to license terms.
 *	Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */

package ej.drawing;

import ej.microui.display.Font;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.RenderableString;

/**
 * microUI-API
 */
public class TransformPainter {

	/**
	 * microUI-API
	 */
	public enum Flip {

		/**
		 * microUI-API
		 */
		FLIP_NONE(0),

		/**
		 * microUI-API
		 */
		FLIP_90(7),

		/**
		 * microUI-API
		 */
		FLIP_180(4),

		/**
		 * microUI-API
		 */
		FLIP_270(2),

		/**
		 * microUI-API
		 */
		FLIP_MIRROR(3),

		/**
		 * microUI-API
		 */
		FLIP_MIRROR_90(1),

		/**
		 * microUI-API
		 */
		FLIP_MIRROR_180(6),

		/**
		 * microUI-API
		 */
		FLIP_MIRROR_270(5);

		/* default */ int val;

		private Flip(int val) {
			this.val = val;
		}
	}

	private TransformPainter() {
		// static methods only
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param flip
	 *            microUI-API
	 */
	public static void drawFlippedImage(GraphicsContext gc, Image image, int x, int y, Flip flip) {
		assert gc != null && image != null && flip != null;
		drawFlippedImage(gc, image, x, y, flip, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param flip
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawFlippedImage(GraphicsContext gc, Image image, int x, int y, Flip flip, int alpha) {
		assert gc != null && image != null && flip != null;
		DrawingPainterNatives.drawFlippedImage(gc.getSNIContext(), image.getSNIContext(), 0, 0, image.getWidth(), image.getHeight(),
				getX(gc, x), getY(gc, y), flip.val, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param xys
	 *            microUI-API
	 */
	public static void drawDeformedImage(GraphicsContext gc, Image image, int x, int y, int[] xys) {
		assert gc != null && image != null && xys != null;

		int count = xys.length;
		if (count != 4 * 2) {
			throw new IllegalArgumentException();
		}

		int[] xy = new int[4 * 2];
		synchronized (xy) {

			for (int i = count; --i >= 0;) {
				xy[i] = getY(gc, xys[i]) + y;
				xy[--i] = getX(gc, xys[i]) + x;
			}

			byte[] workingBuffer = NativesTools.getPolygonWorkingBuffer(4);
			DrawingPainterNatives.drawDeformedImage(gc.getSNIContext(), image.getSNIContext(), xy, workingBuffer);
		}
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 */
	public static void drawRotatedImageBilinear(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle) {
		assert gc != null && image != null;
		DrawingPainterNatives.drawRotatedImageBilinear(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y),
				getX(gc, rx), getY(gc, ry), angle,
				GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawRotatedImageBilinear(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle,
			int alpha) {
		assert gc != null && image != null;
		DrawingPainterNatives.drawRotatedImageBilinear(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y),
				getX(gc, rx), getY(gc, ry), angle, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 */
	public static void drawRotatedImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle) {
		assert gc != null && image != null;
		DrawingPainterNatives.drawRotatedImageNearestNeighbor(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y),
				getX(gc, rx), getY(gc, ry), angle,
				GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawRotatedImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, int rx, int ry,
			float angle,
			int alpha) {
		assert gc != null && image != null;
		DrawingPainterNatives.drawRotatedImageNearestNeighbor(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y),
				getX(gc, rx), getY(gc, ry), angle, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param c
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 */
	public static void drawRotatedCharBilinear(GraphicsContext gc, Font font, char c, int x, int y,
			int rx, int ry, float angle) {
		assert gc != null && font != null;
		drawRotatedCharBilinear(gc, font, c, x, y, rx, ry, angle, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param c
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawRotatedCharBilinear(GraphicsContext gc, Font font, char c, int x, int y,
			int rx, int ry, float angle, int alpha) {
		assert gc != null && font != null;
		byte[] fontData = font.getSNIContext();
		byte[] dest = gc.getSNIContext();
		int dx = getX(gc, x);
		int dy = getY(gc, y);
		int xRotation = getX(gc, rx);
		int yRotation = getY(gc, ry);

		DrawingPainterNatives.drawCharWithRotationBilinear(dest, c, fontData, dx, dy, xRotation, yRotation, angle, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param c
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 */
	public static void drawRotatedCharNearestNeighbor(GraphicsContext gc, Font font, char c, int x,
			int y, int rx, int ry, float angle) {
		assert gc != null && font != null;
		drawRotatedCharNearestNeighbor(gc, font, c, x, y, rx, ry, angle, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param c
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param rx
	 *            microUI-API
	 * @param ry
	 *            microUI-API
	 * @param angle
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawRotatedCharNearestNeighbor(GraphicsContext gc, Font font, char c, int x,
			int y, int rx, int ry, float angle, int alpha) {
		assert gc != null && font != null;
		byte[] fontData = font.getSNIContext();
		byte[] dest = gc.getSNIContext();
		int dx = getX(gc, x);
		int dy = getY(gc, y);
		int xRotation = getX(gc, rx);
		int yRotation = getY(gc, ry);

		DrawingPainterNatives.drawCharWithRotationNearestNeighbor(dest, c, fontData, dx, dy, xRotation, yRotation, angle, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledImageBilinear(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY) {
		assert gc != null && image != null;
		checkScaleFactors(factorX, factorY);
		DrawingPainterNatives.drawScaledImageBilinear(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y), factorX, factorY, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawScaledImageBilinear(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY, int alpha) {
		assert gc != null && image != null;
		checkScaleFactors(factorX, factorY);
		DrawingPainterNatives.drawScaledImageBilinear(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y), factorX, factorY, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY) {
		assert gc != null && image != null;
		checkScaleFactors(factorX, factorY);
		DrawingPainterNatives.drawScaledImageNearestNeighbor(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y), factorX, factorY, GraphicsContext.OPAQUE);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param image
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 * @param alpha
	 *            microUI-API
	 */
	public static void drawScaledImageNearestNeighbor(GraphicsContext gc, Image image, int x, int y, float factorX,
			float factorY, int alpha) {
		assert gc != null && image != null;
		checkScaleFactors(factorX, factorY);
		DrawingPainterNatives.drawScaledImageNearestNeighbor(gc.getSNIContext(), image.getSNIContext(), getX(gc, x),
				getY(gc, y), factorX, factorY, alpha);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param character
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledCharBilinear(GraphicsContext gc, char character, Font font, int x, int y, float factorX,
			float factorY) {
		assert gc != null && font != null;
		checkScaleFactors(factorX, factorY);
		DrawingPainterNatives.drawScaledStringBilinear(gc.getSNIContext(), new char[] { character }, 1,
				font.getSNIContext(), getX(gc, x), getY(gc, y), factorX, factorY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param string
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledStringBilinear(GraphicsContext gc, String string, Font font, int x, int y,
			float factorX,
			float factorY) {
		assert gc != null && string != null && font != null;
		checkScaleFactors(factorX, factorY);
		char[] chars = string.toCharArray();
		DrawingPainterNatives.drawScaledStringBilinear(gc.getSNIContext(), chars, chars.length, font.getSNIContext(),
				getX(gc, x), getY(gc, y), factorX, factorY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param string
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 * @param font
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledSubstringBilinear(GraphicsContext gc, String string, int offset, int length, Font font,
			int x,
			int y, float factorX, float factorY) {
		assert gc != null && string != null && font != null;
		checkScaleFactors(factorX, factorY);
		checkSubstringBounds(string, offset, length);
		char[] chars = new char[length];
		string.getChars(offset, offset + length - 1, chars, 0);
		DrawingPainterNatives.drawScaledStringBilinear(gc.getSNIContext(), chars, length, font.getSNIContext(),
				getX(gc, x), getY(gc, y), factorX, factorY);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param string
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param factorX
	 *            microUI-API
	 * @param factorY
	 *            microUI-API
	 */
	public static void drawScaledRenderableStringBilinear(GraphicsContext gc, RenderableString string, int x, int y,
			float factorX, float factorY) {
		assert gc != null && string != null;
		checkScaleFactors(factorX, factorY);
		char[] chars = string.getCharArray();
		DrawingPainterNatives.drawScaledRenderableStringBilinear(gc.getSNIContext(), chars, chars.length,
				string.getFont().getSNIContext(), string.getWidth(), string.getSNIContext(), getX(gc, x), getY(gc, y), factorX,
				factorY);
	}

	private static void checkSubstringBounds(String str, int offset, int len) {
		int sl = str.length(); // can throw NPE
		if (offset < 0 || len < 0 || offset >= sl || offset + len > sl) {
			throw new StringIndexOutOfBoundsException();
		}
	}

	private static void checkScaleFactors(float factorX, float factorY) {
		if (factorX < 0.0f || factorY < 0.0f) {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * @return the given coordinate shifted with user translation AND {@link GraphicsContext} translation
	 */
	private static int getX(GraphicsContext gc, int x) {
		return x + gc.getTranslationX();
	}

	/**
	 * @return the given coordinate shifted with user translation AND {@link GraphicsContext} translation
	 */
	private static int getY(GraphicsContext gc, int y) {
		return y + gc.getTranslationY();
	}
}
