/*
 * Java
 *
 * Copyright 2020-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.drawing;

import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;

/**
 * microUI-API
 */
public class ShapePainter {

	/**
	 * microUI-API
	 */
	public enum Cap {
		/**
		 * microUI-API
		 */
		NONE(0),

		/**
		 * microUI-API
		 */
		ROUNDED(1),

		/**
		 * microUI-API
		 */
		PERPENDICULAR(2),

		/**
		 * microUI-API
		 */
		HORIZONTAL(3),

		/**
		 * microUI-API
		 */
		VERTICAL(4);

		private final int val;

		private Cap(int val) {
			this.val = val;
		}
	}

	private ShapePainter() {
		// static methods only
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 * @param fade
	 *            microUI-API
	 */
	public static void drawThickFadedPoint(GraphicsContext gc, int x, int y, int thickness, int fade) {
		assert gc != null;
		if (thickness < 0 || fade < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickFadedPoint(gc.getSNIContext(), x + translateX, y + translateY, thickness, fade);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x1
	 *            microUI-API
	 * @param startY
	 *            microUI-API
	 * @param endX
	 *            microUI-API
	 * @param endY
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 */
	public static void drawThickLine(GraphicsContext gc, int x1, int startY, int endX, int endY, int thickness) {
		assert gc != null;
		if (thickness < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickLine(gc.getSNIContext(), x1 + translateX, startY + translateY, endX + translateX,
				endY + translateY, thickness);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param startX
	 *            microUI-API
	 * @param startY
	 *            microUI-API
	 * @param endX
	 *            microUI-API
	 * @param endY
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 * @param fade
	 *            microUI-API
	 * @param startCap
	 *            microUI-API
	 * @param endCap
	 *            microUI-API
	 */
	public static void drawThickFadedLine(GraphicsContext gc, int startX, int startY, int endX, int endY, int thickness, int fade,
			Cap startCap,
			Cap endCap) {
		assert gc != null && startCap != null && endCap != null;
		if (thickness < 0 || fade < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickFadedLine(gc.getSNIContext(), startX + translateX, startY + translateY, endX + translateX,
				endY + translateY, thickness, fade, startCap.val, endCap.val /* not in api yet */);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 */
	public static void drawThickCircleArc(GraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle,
			int thickness) {
		assert gc != null;
		if (thickness < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickCircleArc(gc.getSNIContext(), translateX + x, translateY + y, diameter,
				startAngle, arcAngle, thickness);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param startAngle
	 *            microUI-API
	 * @param arcAngle
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 * @param fade
	 *            microUI-API
	 * @param startCap
	 *            microUI-API
	 * @param endCap
	 *            microUI-API
	 */
	public static void drawThickFadedCircleArc(GraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness, int fade, Cap startCap, Cap endCap) {
		assert gc != null && startCap != null && endCap != null;
		if (thickness < 0 || fade < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickFadedCircleArc(gc.getSNIContext(), translateX + x, translateY + y, diameter,
				startAngle, arcAngle, thickness, fade, startCap.val, endCap.val /* not in api yet */);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-APImicroUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 */
	public static void drawThickCircle(GraphicsContext gc, int x, int y, int diameter, int thickness) {
		assert gc != null;
		if (thickness < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickCircle(gc.getSNIContext(), translateX + x, translateY + y, diameter, thickness);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param diameter
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 * @param fade
	 *            microUI-API
	 */
	public static void drawThickFadedCircle(GraphicsContext gc, int x, int y, int diameter, int thickness, int fade) {
		assert gc != null;
		if (thickness < 0 || fade < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickFadedCircle(gc.getSNIContext(), translateX + x, translateY + y, diameter,
				thickness, fade);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 */
	public static void drawThickEllipse(GraphicsContext gc, int x, int y, int width, int height, int thickness) {
		assert gc != null;
		if (thickness < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickEllipse(gc.getSNIContext(), translateX + x, translateY + y, width, height,
				thickness);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param x
	 *            microUI-API
	 * @param y
	 *            microUI-API
	 * @param width
	 *            microUI-API
	 * @param height
	 *            microUI-API
	 * @param thickness
	 *            microUI-API
	 * @param fade
	 *            microUI-API
	 */
	public static void drawThickFadedEllipse(GraphicsContext gc, int x, int y, int width, int height, int thickness,
			int fade) {
		assert gc != null;
		if (thickness < 0 || fade < 0) {
			throw new IllegalArgumentException();
		}
		int translateX = gc.getTranslationX();
		int translateY = gc.getTranslationY();
		DrawingPainterNatives.drawThickFadedEllipse(gc.getSNIContext(), translateX + x, translateY + y, width, height,
				thickness, fade);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xys
	 *            microUI-API
	 */
	public static void drawPolygon(GraphicsContext gc, int[] xys) {
		drawPolygon(gc, xys, 0, xys.length);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xys
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 */
	public static void drawPolygon(GraphicsContext gc, int[] xys, int offset, int length) {
		assert gc != null && xys != null;
		xys = check(gc, xys, offset, length);
		if (xys != null) {
			DrawingPainterNatives.drawPolygon(gc.getSNIContext(), xys, xys.length);
		}
		// else: nothing to draw
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xys
	 *            microUI-API
	 */
	public static void fillPolygon(GraphicsContext gc, int[] xys) {
		assert gc != null && xys != null;
		fillPolygon(gc, xys, 0, xys.length);
	}

	/**
	 * microUI-API
	 *
	 * @param gc
	 *            microUI-API
	 * @param xys
	 *            microUI-API
	 * @param offset
	 *            microUI-API
	 * @param length
	 *            microUI-API
	 */
	public static void fillPolygon(GraphicsContext gc, int[] xys, int offset, int length) {
		assert gc != null && xys != null;
		xys = check(gc, xys, offset, length);
		if (xys != null) {
			byte[] workingBuffer = NativesTools.getPolygonWorkingBuffer(length / 2);
			DrawingPainterNatives.fillPolygon(gc.getSNIContext(), xys, xys.length, workingBuffer);
		}
		// else: nothing to draw
	}

	/**
	 * Check the xys array validity. if array is invalid, throw the right exception. if the array is smaller than 6,
	 * draw the pixel, the line or nothing and returns false. otherwise returns true;<br>
	 */
	private static @Nullable int[] check(GraphicsContext gc, int[] xys, int offset, int length) {

		checkArrayBounds(xys.length, offset, length);

		if ((length & 1) != 0) {
			throw new IllegalArgumentException();
		}

		if (length == 0) {
			// nothing to draw !
			return null;
		} else if (length == 2) {
			// draw a pixel
			Painter.writePixel(gc, xys[0], xys[1]);
			return null;
		} else if (length == 4) {
			// draw a line
			Painter.drawLine(gc, xys[0], xys[1], xys[2], xys[3]);
			return null;
		}

		// copy to the working array (do not modify caller array)
		int[] out = new int[length];
		System.arraycopy(xys, offset, out, 0, length);

		// apply translation
		int tx = gc.getTranslationX();
		int ty = gc.getTranslationY();
		if (tx != 0 || ty != 0) {
			for (int i = 0; i < length; i += 2) {
				out[i] += tx;
				out[i + 1] += ty;
			}
		}

		// drawing has to be done
		return out;
	}

	private static void checkArrayBounds(int arrayLength, int off, int len) {
		int offPlusLen = off + len;
		if (off < 0 || len < 0 || offPlusLen > arrayLength || offPlusLen < 0) {
			throw new ArrayIndexOutOfBoundsException();
		}
	}
}
