/*
 * Java
 *
 * Copyright 2020-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.drawing;

import ej.drawing.ShapePainter.Cap;
import ej.drawing.TransformPainter.Flip;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;

/**
 * LLDW_PAINTER_impl.h
 */
public class DrawingPainterNatives {

	private DrawingPainterNatives() {
		// static class
	}

	/**
	 * LLDW_PAINTER_IMPL_drawAAPoint
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link AntiAliasedShape#getThickness()}
	 * @param fade
	 *            {@link AntiAliasedShape#getFade()}
	 * @param x
	 *            the x coordinate of the point
	 * @param y
	 *            the y coordinate of the point
	 */
	public static native void drawThickFadedPoint(byte[] gc, int x, int y, int thickness, int fade);

	/**
	 * LLDW_PAINTER_IMPL_drawAALine
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link AntiAliasedShape#getThickness()}
	 * @param fade
	 *            {@link AntiAliasedShape#getFade()}
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param capStart
	 *            see {@link AntiAliasedShape#setStartCap(Cap)}
	 * @param capEnd
	 *            see {@link AntiAliasedShape#setEndCap(Cap)}
	 */
	public static native void drawThickFadedLine(byte[] gce, int x1, int y1, int x2, int y2, int thickness, int fade,
			int capStart, int capEnd);

	/**
	 * LLDW_PAINTER_IMPL_drawAACircleArc
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link AntiAliasedShape#getThickness()}
	 * @param fade
	 *            {@link AntiAliasedShape#getFade()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is drawn
	 * @param diameter
	 *            the diameter of the arc to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from <code>startAngle</code>
	 * @param capStart
	 *            see {@link AntiAliasedShape#setStartCap(Cap)}
	 * @param capEnd
	 *            see {@link AntiAliasedShape#setEndCap(Cap)}
	 */
	public static native void drawThickFadedCircleArc(byte[] gc, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness, int fade, int capStart, int capEnd);

	/**
	 * LLDW_PAINTER_IMPL_drawAAEllipse
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link AntiAliasedShape#getThickness()}
	 * @param fade
	 *            {@link AntiAliasedShape#getFade()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 */
	public static native void drawThickFadedEllipse(byte[] gc, int x, int y, int width, int height, int thickness,
			int fade);

	/**
	 * LLDW_PAINTER_IMPL_drawAACircle
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link AntiAliasedShape#getThickness()}
	 * @param fade
	 *            {@link AntiAliasedShape#getFade()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 */
	public static native void drawThickFadedCircle(byte[] gc, int x, int y, int diameter, int thickness, int fade);

	/**
	 * LLDW_PAINTER_IMPL_drawThickLine
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link ShapePainter#getThickness()}
	 * @param x1
	 *            the x coordinate of the start of the line.
	 * @param y1
	 *            the y coordinate of the start of the line.
	 * @param x2
	 *            the x coordinate of the end of the line.
	 * @param y2
	 *            the y coordinate of the end of the line.
	 */
	public static native void drawThickLine(byte[] gc, int x1, int y1, int x2, int y2, int thickness);

	/**
	 * LLDW_PAINTER_IMPL_drawThickCircleArc
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link ShapePainter#getThickness()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the arc is drawn.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the arc is drawn.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw.
	 * @param arcAngle
	 *            the angular extent of the arc from <code>startAngle</code>.
	 */
	public static native void drawThickCircleArc(byte[] gc, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness);

	/**
	 * LLDW_PAINTER_IMPL_drawThickCircle
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link ShapePainter#getThickness()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param diameter
	 *            the diameter of the circle to draw.
	 */
	public static native void drawThickCircle(byte[] gc, int x, int y, int diameter, int thickness);

	/**
	 * LLDW_PAINTER_IMPL_drawThickEllipse
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param thickness
	 *            {@link ShapePainter#getThickness()}
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the circle is drawn.
	 * @param diameter
	 *            the diameter of the circle to draw.
	 */
	public static native void drawThickEllipse(byte[] gc, int x, int y, int width, int height, int thickness);

	/**
	 * LLDW_PAINTER_IMPL_drawDeformedImage
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param xys
	 *            4 corners new position of left-top, right-top, right-bottom and left-bottom corners.
	 * @param workingBuffer
	 *            software algoithm working buffer
	 */
	public static native void drawDeformedImage(byte[] destGC, byte[] srcImage, int[] xys, byte[] workingBuffer);

	/**
	 * LLDW_PAINTER_IMPL_scaleImageNearestNeighbor
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param x
	 *            the x coordinate of upper-left corner
	 * @param y
	 *            the y coordinate of upper-left corner
	 * @param factorX
	 *            {@link Scale#getFactorX()}
	 * @param factorY
	 *            {@link Scale#getFactorY()}
	 * @param alpha
	 *            {@link Scale#getAlpha()}
	 * @param mirror
	 *            {@link Scale#isMirrored()}
	 */
	public static native void drawScaledImageNearestNeighbor(byte[] destGC, byte[] srcImage, int x, int y,
			float factorX, float factorY, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_scaleImageBilinear
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param x
	 *            the x coordinate of destination upper-left corner
	 * @param y
	 *            the y coordinate of destination upper-left corner
	 * @param factorX
	 *            {@link Scale#getFactorX()}
	 * @param factorY
	 *            {@link Scale#getFactorY()}
	 * @param alpha
	 *            {@link Scale#getAlpha()}
	 * @param mirror
	 *            {@link Scale#isMirrored()}
	 */
	public static native void drawScaledImageBilinear(byte[] destGC, byte[] srcImage, int x, int y, float factorX,
			float factorY, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_flipImage
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param xSrc
	 *            the x coordinate of source upper-left corner
	 * @param ySrc
	 *            the y coordinate of source upper-left corner
	 * @param width
	 *            the region width to flip
	 * @param height
	 *            the region height to flip
	 * @param xDest
	 *            the x coordinate of destination upper-left corner
	 * @param yDest
	 *            the y coordinate of destination upper-left corner
	 * @param transformation
	 *            {@link Flip#getAction()}
	 * @param alpha
	 *            {@link Flip#getAlpha()}
	 */
	public static native void drawFlippedImage(byte[] destGC, byte[] srcImage, int xSrc, int ySrc, int width,
			int height, int xDest, int yDest, int transformation, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_rotateImageBilinear
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param destX
	 *            the x coordinate of destination upper-left corner
	 * @param destY
	 *            the y coordinate of destination upper-left corner
	 * @param xRotation
	 *            {@link Rotate#getRotationX()}
	 * @param yRotation
	 *            {@link Rotate#getRotationX()}
	 * @param angle
	 *            {@link Rotate#getAngle()}
	 * @param alpha
	 *            {@link Rotate#getAlpha()}}
	 * @param mirror
	 *            {@link Rotate#isMirrored()}
	 */
	public static native void drawRotatedImageBilinear(byte[] destGC, byte[] srcImage, int destX, int destY,
			int xRotation, int yRotation, float angle, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_rotateImageNearestNeighbor
	 *
	 * @param destGC
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param srcImage
	 *            source context ({@link Image#getSNIContext()})
	 * @param destX
	 *            the x coordinate of destination upper-left corner
	 * @param destY
	 *            the y coordinate of destination upper-left corner
	 * @param xRotation
	 *            {@link Rotate#getRotationX()}
	 * @param yRotation
	 *            {@link Rotate#getRotationX()}
	 * @param angle
	 *            {@link Rotate#getAngle()}
	 * @param alpha
	 *            {@link Rotate#getAlpha()}}
	 * @param mirror
	 *            {@link Rotate#isMirrored()}
	 */
	public static native void drawRotatedImageNearestNeighbor(byte[] destGC, byte[] srcImage, int destX, int destY,
			int xRotation, int yRotation, float angle, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_rotateCharacterNearestNeighbor
	 *
	 * @param sd
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param c
	 *            character to draw
	 * @param fontData
	 *            font native data
	 * @param destX
	 *            the x coordinate of destination upper-left corner
	 * @param destY
	 *            the y coordinate of destination upper-left corner
	 * @param xRotation
	 *            the horizontal coordinate of the rotation center
	 * @param yRotation
	 *            the vertical coordinate of the rotation center
	 * @param angle
	 *            the rotation angle, in degrees
	 * @param alpha
	 *            the alpha value to apply
	 */
	public static native void drawCharWithRotationNearestNeighbor(byte[] sd, // NOSONAR cannot pass object to a native
			char c, byte[] fontData, int destX, int destY, int xRotation, int yRotation, float angle, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_rotateCharacterBilinear
	 *
	 * @param sd
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param c
	 *            character to draw
	 * @param fontData
	 *            font native data
	 * @param destX
	 *            the x coordinate of destination upper-left corner
	 * @param destY
	 *            the y coordinate of destination upper-left corner
	 * @param xRotation
	 *            the horizontal coordinate of the rotation center
	 * @param yRotation
	 *            the vertical coordinate of the rotation center
	 * @param angle
	 *            the rotation angle, in degrees
	 * @param alpha
	 *            the alpha value to apply
	 */
	public static native void drawCharWithRotationBilinear(byte[] sd, // NOSONAR cannot pass object to a native
			char c, byte[] fontData, int destX, int destY, int xRotation, int yRotation, float angle, int alpha);

	/**
	 * LLDW_PAINTER_IMPL_drawScaledCharsBilinear
	 *
	 *
	 * @param sd
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param chars
	 *            characters to draw
	 * @param length
	 *            the number of characters.
	 * @param font
	 *            the font to use
	 * @param x
	 *            the x coordinate of the string.
	 * @param y
	 *            the y coordinate of the string.
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 */
	public static native void drawScaledStringBilinear(byte[] sd, // NOSONAR cannot pass object to a native
			char[] chars, int length, byte[] font, int x, int y, float factorX, float factorY);

	/**
	 * LLDW_PAINTER_IMPL_drawScaledCharsBilinear
	 *
	 *
	 * @param sd
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param chars
	 *            characters to draw
	 * @param length
	 *            the number of characters.
	 * @param font
	 *            the font to use
	 * @param width
	 *            the string width in pixels when known, 0 otherwise.
	 * @param renderableString
	 *            the renderable string.
	 * @param x
	 *            the x coordinate of the string.
	 * @param y
	 *            the y coordinate of the string.
	 * @param factorX
	 *            the scaling X factor.
	 * @param factorY
	 *            the scaling Y factor.
	 */
	public static native void drawScaledRenderableStringBilinear(byte[] sd, // NOSONAR cannot pass object to a native
			char[] chars, int length, byte[] font, int width, byte[] renderableString, int x, int y, float factorX,
			float factorY);

	/**
	 * LLDW_PAINTER_IMPL_drawPolygon
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @param length
	 *            the array's length of coordinates
	 */
	public static native void drawPolygon(byte[] gc, int[] xys, int length);

	/**
	 * LLDW_PAINTER_IMPL_fillPolygon
	 *
	 * @param gc
	 *            destination context ({@link GraphicsContext#getSNIContext()})
	 * @param xys
	 *            the array of coordinates : x1,y1,......xn,yn.
	 * @param length
	 *            the array's length of coordinates
	 * @param workingBuffer
	 *            software algoithm working buffer
	 */
	public static native void fillPolygon(byte[] gc, int[] xys, int length, byte[] workingBuffer);

}
