/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2019, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.io;

import java.security.Permission;

@SuppressWarnings("serial")
public final class FilePermission extends Permission implements Serializable {

	// NOTE: this implementation tries to do the minimum computation stuff for the regular case
	// <li>permission i created but is not checked
	// <li>there is only one action</li>

	public static final String ACTION_READ 		= "read";
	public static final String ACTION_WRITE 	= "write";
	public static final String ACTION_EXECUTE 	= "execute";
	public static final String ACTION_DELETE 	= "delete";
	public static final String ACTION_READLINK 	= "readlink";

	// ids in order as should be returned by #getActions():read, write, execute, delete, readlink.
	public static final int READ_ID 		= 0;
	public static final int WRITE_ID 		= 1;
	public static final int EXECUTE_ID 		= 2;
	public static final int DELETE_ID 		= 3;
	public static final int READLINK_ID 	= 4;

	public static final int READ_MASK 		= 1<<READ_ID;
	public static final int WRITE_MASK 		= 1<<WRITE_ID;
	public static final int EXECUTE_MASK 	= 1<<EXECUTE_ID;
	public static final int DELETE_MASK 	= 1<<DELETE_ID;
	public static final int READLINK_MASK 	= 1<<READLINK_ID;

	public static final int ACTIONS_OFFSET 	= 0;
	public static final int ACTIONS_SIZE 	= 8;
	public static final int ACTIONS_MASK 	= (1<<ACTIONS_SIZE)-1;

	public static final int NB_ACTIONS_OFFSET 	= ACTIONS_OFFSET+ACTIONS_SIZE;
	public static final int NB_ACTIONS_SIZE 	= 8;
	public static final int NB_ACTIONS_MASK 	= (1<<NB_ACTIONS_SIZE)-1;

	private final int mask;

	// canonicalized dir path. In the case of
	// directories, it is the name "/blah/*" or "/blah/-" without
	// the last character (the "*" or "-").
	private String cpath;

	// static Strings used by init(int mask)
	private static final char RECURSIVE_CHAR = '-';
	private static final char WILD_CHAR = '*';

	// does path indicate a directory? (wildcard or recursive)
	private boolean directory;

	// is it a recursive directory specification?
	private boolean recursive;

	public static void checkFSAttributes() {
		checkPermission(new RuntimePermission("getFileSystemAttributes"));
	}

	private static void checkPermission(Permission p) {
		SecurityManager sm = System.getSecurityManager();
		if(sm != null){
			sm.checkPermission(p);
		}
	}

	public static void checkWrite(String file) {
		checkPermission(new FilePermission(file, (1<<NB_ACTIONS_OFFSET) | (WRITE_MASK<<ACTIONS_OFFSET)));
	}

	public static void checkExec(String file) {
		checkPermission(new FilePermission(file, (1<<NB_ACTIONS_OFFSET) | (EXECUTE_MASK<<ACTIONS_OFFSET)));
	}


	public static void checkDelete(String file) {
		checkPermission(new FilePermission(file, (1<<NB_ACTIONS_OFFSET) | (DELETE_MASK<<ACTIONS_OFFSET)));
	}

	public static void checkRead(String file) {
		SecurityManager sm = System.getSecurityManager();
		if(sm != null){
			sm.checkPermission(new FilePermission(file, (1<<NB_ACTIONS_OFFSET) | (READ_MASK<<ACTIONS_OFFSET)));
		}
	}



	/*
	 * Slow constructor. Must not be used internally in this library.
	 */
	public FilePermission(String path, String actions) {
		super(path);

		if(actions == null){
			throw new IllegalArgumentException();
		}

		// parse actions
		int mask = 0;
		int start = -1; // ptr on the previous comma
		int end = actions.length();
		while(start < end){
			int indexComma = actions.indexOf(',', start+1);
			int actionEnd = indexComma == -1 ? end : indexComma;
			String action = actions.substring(start+1, actionEnd).toLowerCase();
			if(action.equals(ACTION_READ)){
				mask |= READ_MASK;
			}
			else if(action.equals(ACTION_WRITE)){
				mask |= WRITE_MASK;
			}
			else if(action.equals(ACTION_EXECUTE)){
				mask |= EXECUTE_MASK;
			}
			else if(action.equals(ACTION_DELETE)){
				mask |= DELETE_MASK;
			}
			else if(action.equals(ACTION_READLINK)){
				mask |= READLINK_MASK;
			}
			else{
				throw new IllegalArgumentException(action);
			}
			start = actionEnd;
		}

		int nbActions = 0;
		int tmpMask = mask;
		while(tmpMask != 0){
			if((tmpMask & 1) != 0){
				++nbActions;
			}
			tmpMask >>>= 1;
		}

		if(nbActions == 0){
			throw new IllegalArgumentException();
		}

		this.mask = (nbActions << NB_ACTIONS_OFFSET) | (mask << ACTIONS_OFFSET);


		cpath = path;
		if (path.equals("<<ALL FILES>>")) {
			directory = true;
			recursive = true;
			cpath = "";
			return;
		}

		// store only the canonical cpath if possible
		cpath = getCanonicalPath(cpath);
		path = cpath;

		int len = path.length();
		char last = ((len > 0) ? path.charAt(len - 1) : 0);

		if (last == RECURSIVE_CHAR &&
				path.charAt(len - 2) == File.separatorChar) {
			directory = true;
			recursive = true;
			cpath = path.substring(0, --len);
		} else if (last == WILD_CHAR &&
				path.charAt(len - 2) == File.separatorChar) {
			directory = true;
			//recursive = false;
			cpath = path.substring(0, --len);
		} else {
			// overkill since they are initialized to false, but
			// commented out here to remind us...
			//directory = false;
			//recursive = false;
		}

		// XXX: at this point the path should be absolute. die if it isn't?
	}

	private String getCanonicalPath(String cpath2) {
		try {
			String path = cpath2;
			if (path.endsWith("*")) {
				// call getCanonicalPath with a path with wildcard character
				// replaced to avoid calling it with paths that are
				// intended to match all entries in a directory
				path = path.substring(0, path.length()-1) + "-";
				path = new File(path).getCanonicalPath();
				return path.substring(0, path.length()-1) + "*";
			} else {
				return new File(path).getCanonicalPath();
			}
		} catch (IOException | SecurityException e) {
			return cpath2;
		}
	}

	protected FilePermission(String path, int mask) {
		super(path);
		this.mask = mask;
	}

	@Override
	public boolean implies(Permission p) {
		if (!(p instanceof FilePermission)) {
			return false;
		}

		FilePermission that = (FilePermission) p;

		// we get the effective mask. i.e., the "and" of this and that.
		// They must be equal to that.mask for implies to return true.

		int thisMask = (this.mask>>>ACTIONS_OFFSET) & ACTIONS_MASK;
		int thatMask = (that.mask>>>ACTIONS_OFFSET) & ACTIONS_MASK;
		return ((thisMask & thatMask) == thatMask) && impliesIgnoreMask(that);
	}

	/**
	 * Checks if the Permission's actions are a proper subset of the
	 * this object's actions. Returns the effective mask iff the
	 * this FilePermission's path also implies that FilePermission's path.
	 *
	 * @param that the FilePermission to check against.
	 * @return the effective mask
	 */
	boolean impliesIgnoreMask(FilePermission that) {
		String thisCPath = this.cpath;
		String thatCPath = that.cpath;

		if (this.directory) {
			if (this.recursive) {
				// make sure that.path is longer then path so
				// something like /foo/- does not imply /foo
				if (that.directory) {
					return (thatCPath.length() >= thisCPath.length()) &&
							thatCPath.startsWith(thisCPath);
				}  else {
					return ((thatCPath.length() > thisCPath.length()) &&
							thatCPath.startsWith(thisCPath));
				}
			} else {
				if (that.directory) {
					// if the permission passed in is a directory
					// specification, make sure that a non-recursive
					// permission (i.e., this object) can't imply a recursive
					// permission.
					if (that.recursive) {
						return false;
					} else {
						return (thisCPath.equals(thatCPath));
					}
				} else {
					int last = thatCPath.lastIndexOf(File.separatorChar);
					if (last == -1) {
						return false;
					} else {
						// thisCPath.equals(thatCPath.substring(0, last+1));
						// Use regionMatches to avoid creating new string
						return (thisCPath.length() == (last + 1)) &&
								thisCPath.regionMatches(0, thatCPath, 0, last+1);
					}
				}
			}
		} else if (that.directory) {
			// if this is NOT recursive/wildcarded,
			// do not let it imply a recursive/wildcarded permission
			return false;
		} else {
			return (thisCPath.equals(thatCPath));
		}
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == this) {
			return true;
		}

		if (! (obj instanceof FilePermission)) {
			return false;
		}

		FilePermission that = (FilePermission) obj;

		return (this.mask == that.mask) &&
				this.cpath.equals(that.cpath) &&
				(this.directory == that.directory) &&
				(this.recursive == that.recursive);
	}

	@Override
	public int hashCode() {
		return 0;
	}

	@Override
	public String getActions() {
		// Order: read, write, execute, delete, readlink.
		int nbActions = (mask>>>NB_ACTIONS_OFFSET) & NB_ACTIONS_MASK;
		int tmpMask = (mask>>>ACTIONS_OFFSET) & ACTIONS_MASK;

		if(nbActions == 1){
			// single action
			int actionId = 0;
			while(true){
				if((tmpMask & 1) != 0){
					return getAction(actionId);
				}
				++actionId;
				tmpMask >>>= 1;
			}
		}
		else{
			// multiple actions
			StringBuilder sb = new StringBuilder();
			int actionId = 0;
			boolean first = true;
			while(tmpMask != 0){
				if((tmpMask & 1) != 0){
					if(first){
						first = false;
					}
					else{
						sb.append(',');
					}
					sb.append(getAction(actionId));
				}
				++actionId;
				tmpMask >>>= 1;
			}
			return sb.toString();
		}
	}

	private static String getAction(int actionId){
		switch(actionId){
		case READ_ID:		return ACTION_READ;
		case WRITE_ID:		return ACTION_WRITE;
		case EXECUTE_ID:	return ACTION_EXECUTE;
		case DELETE_ID:		return ACTION_DELETE;
		case READLINK_ID:	return ACTION_READLINK;
		default: throw new AssertionError();
		}
	}

}

