/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.io;

import com.is2t.java.io.FileSystem;
import com.is2t.tools.ArrayTools;

import ej.lang.Resource;
import ej.lang.ResourceManager;

/**
 * A file output stream is an output stream for writing data to a
 * <code>File</code>. Whether or not
 * a file is available or may be created depends upon the underlying
 * platform.  Some platforms, in particular, allow a file to be opened
 * for writing by only one <tt>FileOutputStream</tt> (or other
 * file-writing object) at a time.  In such situations the constructors in
 * this class will fail if the file involved is already open.
 *
 * <p><code>FileOutputStream</code> is meant for writing streams of raw bytes
 * such as image data. For writing streams of characters, consider using
 * <code>FileWriter</code>.
 *
 * @author  Arthur van Hoff
 * @see     java.io.File
 * @see     java.io.FileInputStream
 * @since   JDK1.0
 */
public
class FileOutputStream extends OutputStream implements Resource
{
	// WARNING:
	// All the methods that use the fd field are synchronized.
	// This prevents a close during an access to the native object (referenced
	// by fd).
	// Another issue is that the fd may be recycled after the channel is closed.
	// To
	// avoid conflict (we are reading in a fd that has been closed and recycled
	// while
	// we are reading), just keep all this stuff synchronized.

	/**
	 * File Descriptor.
	 * <p>
	 * Initialized on creation. Set to -1 when closed.
	 */
	private int fd;

	/**
	 * Creates a file output stream to write to the file with the
	 * specified name.
	 * <p>
	 * First, if there is a security manager, its <code>checkWrite</code>
	 * method is called with <code>name</code> as its argument.
	 * <p>
	 * If the file exists but is a directory rather than a regular file, does
	 * not exist but cannot be created, or cannot be opened for any other
	 * reason then a <code>FileNotFoundException</code> is thrown.
	 *
	 * @param      name   the system-dependent filename
	 * @exception  FileNotFoundException  if the file exists but is a directory
	 *                   rather than a regular file, does not exist but cannot
	 *                   be created, or cannot be opened for any other reason
	 * @exception  SecurityException  if a security manager exists and its
	 *               <code>checkWrite</code> method denies write access
	 *               to the file.
	 */
	public FileOutputStream(String name) throws FileNotFoundException {
		this(name != null ? new File(name) : null, false);
	}

	/**
	 * Creates a file output stream to write to the file with the specified
	 * name.  If the second argument is <code>true</code>, then
	 * bytes will be written to the end of the file rather than the beginning.
	 * <p>
	 * First, if there is a security manager, its <code>checkWrite</code>
	 * method is called with <code>name</code> as its argument.
	 * <p>
	 * If the file exists but is a directory rather than a regular file, does
	 * not exist but cannot be created, or cannot be opened for any other
	 * reason then a <code>FileNotFoundException</code> is thrown.
	 *
	 * @param     name        the system-dependent file name
	 * @param     append      if <code>true</code>, then bytes will be written
	 *                   to the end of the file rather than the beginning
	 * @exception  FileNotFoundException  if the file exists but is a directory
	 *                   rather than a regular file, does not exist but cannot
	 *                   be created, or cannot be opened for any other reason.
	 * @exception  SecurityException  if a security manager exists and its
	 *               <code>checkWrite</code> method denies write access
	 *               to the file.
	 * @since     JDK1.1
	 */
	public FileOutputStream(String name, boolean append)
			throws FileNotFoundException
	{
		this(name != null ? new File(name) : null, append);
	}

	/**
	 * Creates a file output stream to write to the file represented by
	 * the specified <code>File</code> object.
	 * <p>
	 * First, if there is a security manager, its <code>checkWrite</code>
	 * method is called with the path represented by the <code>file</code>
	 * argument as its argument.
	 * <p>
	 * If the file exists but is a directory rather than a regular file, does
	 * not exist but cannot be created, or cannot be opened for any other
	 * reason then a <code>FileNotFoundException</code> is thrown.
	 *
	 * @param      file               the file to be opened for writing.
	 * @exception  FileNotFoundException  if the file exists but is a directory
	 *                   rather than a regular file, does not exist but cannot
	 *                   be created, or cannot be opened for any other reason
	 * @exception  SecurityException  if a security manager exists and its
	 *               <code>checkWrite</code> method denies write access
	 *               to the file.
	 * @see        java.io.File#getPath()
	 * @see        java.lang.SecurityException
	 */
	public FileOutputStream(File file) throws FileNotFoundException {
		this(file, false);
	}

	/**
	 * Creates a file output stream to write to the file represented by
	 * the specified <code>File</code> object. If the second argument is
	 * <code>true</code>, then bytes will be written to the end of the file
	 * rather than the beginning.
	 * <p>
	 * First, if there is a security manager, its <code>checkWrite</code>
	 * method is called with the path represented by the <code>file</code>
	 * argument as its argument.
	 * <p>
	 * If the file exists but is a directory rather than a regular file, does
	 * not exist but cannot be created, or cannot be opened for any other
	 * reason then a <code>FileNotFoundException</code> is thrown.
	 *
	 * @param      file               the file to be opened for writing.
	 * @param     append      if <code>true</code>, then bytes will be written
	 *                   to the end of the file rather than the beginning
	 * @exception  FileNotFoundException  if the file exists but is a directory
	 *                   rather than a regular file, does not exist but cannot
	 *                   be created, or cannot be opened for any other reason
	 * @exception  SecurityException  if a security manager exists and its
	 *               <code>checkWrite</code> method denies write access
	 *               to the file.
	 * @see        java.io.File#getPath()
	 * @see        java.lang.SecurityException
	 * @since 1.4
	 */
	public FileOutputStream(File file, boolean append)
			throws FileNotFoundException
	{
		String name = file.path;
		FilePermission.checkWrite(name);
		if (file.isInvalid()) {
			throw new FileNotFoundException("Invalid file path");
		}
		try {
			char mode = append ? FileSystem.FILE_MODE_APPEND : FileSystem.FILE_MODE_WRITE;
			this.fd = FileSystem.openNative(file.getAbsolutePathCString(), mode);

			// new file resource created, add it to the resource manager
			ResourceManager rm = ResourceManager.getResourceManager();
			if (rm != null) {
				rm.resourceCreated(this);
			}
		} catch (IOException ioe) {
			FileNotFoundException fnfe = new FileNotFoundException(file.getPath());
			fnfe.initCause(ioe);
			throw fnfe;
		}
	}

	/**
	 * Writes the specified byte to this file output stream. Implements
	 * the <code>write</code> method of <code>OutputStream</code>.
	 *
	 * @param      b   the byte to be written.
	 * @exception  IOException  if an I/O error occurs.
	 */
	@Override
	public synchronized void write(int b) throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			File.throwClosedException();
		}

		FileSystem.writeByteNative(fd, b);
	}

	/**
	 * Writes <code>b.length</code> bytes from the specified byte array
	 * to this file output stream.
	 *
	 * @param      b   the data.
	 * @exception  IOException  if an I/O error occurs.
	 */
	@Override
	public void write(byte[] b) throws IOException {
		this.write(b, 0, b.length);
	}

	/**
	 * Writes <code>len</code> bytes from the specified byte array
	 * starting at offset <code>off</code> to this file output stream.
	 *
	 * @param      b     the data.
	 * @param      off   the start offset in the data.
	 * @param      len   the number of bytes to write.
	 * @exception  IOException  if an I/O error occurs.
	 */
	@Override
	public synchronized void write(byte b[], int off, int len) throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			File.throwClosedException();
		}

		ArrayTools.checkBounds(b, off, len);

		// write until there remains no data to write.
		while (len > 0) {

			int currentWriteSize = FileSystem.writeNative(fd, b, off, len);

			len -= currentWriteSize;
			off += currentWriteSize;
		}
	}

	/**
	 * Closes this file output stream and releases any system resources
	 * associated with this stream. This file output stream may no longer
	 * be used for writing bytes.
	 *
	 * <p> If this stream has an associated channel then the channel is closed
	 * as well.
	 *
	 * @exception  IOException  if an I/O error occurs.
	 *
	 * @revised 1.4
	 * @spec JSR-51
	 */
	@Override
	public void close() throws IOException {
		int fd;
		synchronized (this) {
			fd = this.fd;
			if (fd == -1) {
				return; // Already closed
			}
			this.fd = -1;
		}

		FileSystem.closeNative(fd);
		// file resource closed, remove it to the resource manager
		ResourceManager rm = ResourceManager.getResourceManager();
		if (rm != null) {
			rm.resourceReclaimed(this);
		}
	}

	@Override
	public Object getSource() {
		return this;
	}

	@Override
	public synchronized void reclaim() {
		try {
			close();
		} catch (IOException e) {
		}
	}

	@Override
	public void flush() throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			return; // Already closed
		}

		FileSystem.flushNative(fd);
	}

}
