/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2023, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.io;

import com.is2t.java.io.FileSystem;
import com.is2t.tools.ArrayTools;

import ej.lang.Resource;
import ej.lang.ResourceManager;

/**
 * A <code>FileInputStream</code> obtains input bytes
 * from a file in a file system. What files
 * are  available depends on the host environment.
 *
 * <p><code>FileInputStream</code> is meant for reading streams of raw bytes
 * such as image data. For reading streams of characters, consider using
 * <code>FileReader</code>.
 *
 * @author Arthur van Hoff
 * @see java.io.File
 * @see java.io.FileOutputStream
 * @since JDK1.0
 */
public
class FileInputStream extends InputStream implements Resource
{
	// WARNING:
	// All the methods that use the fd field are synchronized.
	// This prevents a close during an access to the native object (referenced
	// by fd).
	// Another issue is that the fd may be recycled after the channel is closed.
	// To
	// avoid conflict (we are reading in a fd that has been closed and recycled
	// while
	// we are reading), just keep all this stuff synchronized.

	/**
	 * File Descriptor.
	 * <p>
	 * Initialized on creation. Set to -1 when closed.
	 */
	private int fd;

	/**
	 * sync close and reclaim. close and reclaim can be executed in parallel thread in KF stop applications
	 */
	private final Object closesync = new Object();

	/**
	 * Creates a <code>FileInputStream</code> by
	 * opening a connection to an actual file,
	 * the file named by the path name <code>name</code>
	 * in the file system.
	 * <p>
	 * First, if there is a security
	 * manager, its <code>checkRead</code> method
	 * is called with the <code>name</code> argument
	 * as its argument.
	 * <p>
	 * If the named file does not exist, is a directory rather than a regular
	 * file, or for some other reason cannot be opened for reading then a
	 * <code>FileNotFoundException</code> is thrown.
	 *
	 * @param name the system-dependent file name.
	 * @exception  FileNotFoundException  if the file does not exist,
	 *                   is a directory rather than a regular file,
	 *                   or for some other reason cannot be opened for
	 *                   reading.
	 * @exception SecurityException     if a security manager exists and its
	 *               <code>checkRead</code> method denies read access
	 *               to the file.
	 */
	public FileInputStream(String name) throws FileNotFoundException {
		this(name != null ? new File(name) : null);
	}

	/**
	 * Creates a <code>FileInputStream</code> by
	 * opening a connection to an actual file,
	 * the file named by the <code>File</code>
	 * object <code>file</code> in the file system.
	 * <p>
	 * First, if there is a security manager,
	 * its <code>checkRead</code> method  is called
	 * with the path represented by the <code>file</code>
	 * argument as its argument.
	 * <p>
	 * If the named file does not exist, is a directory rather than a regular
	 * file, or for some other reason cannot be opened for reading then a
	 * <code>FileNotFoundException</code> is thrown.
	 *
	 * @param file the file to be opened for reading.
	 * @exception  FileNotFoundException  if the file does not exist,
	 *                   is a directory rather than a regular file,
	 *                   or for some other reason cannot be opened for
	 *                   reading.
	 * @exception SecurityException     if a security manager exists and its
	 *               <code>checkRead</code> method denies read access to the file.
	 * @see java.io.File#getPath()
	 */
	public FileInputStream(File file) throws FileNotFoundException {
		String name = file.path;
		FilePermission.checkRead(name);
		if (file.isInvalid()) {
			throw new FileNotFoundException("Invalid file path");
		}
		try {
			this.fd = FileSystem.openNative(file.getAbsolutePathCString(), FileSystem.FILE_MODE_READ);

			// new file resource created, add it to the resource manager
			ResourceManager rm = ResourceManager.getResourceManager();
			if (rm != null) {
				rm.resourceCreated(this);
			}

		} catch (IOException ioe) {
			FileNotFoundException fnfe = new FileNotFoundException(file.getPath());
			fnfe.initCause(ioe);
			throw fnfe;
		}
	}

	/**
	 * Reads a byte of data from this input stream. This method blocks
	 * if no input is yet available.
	 *
	 * @return     the next byte of data, or <code>-1</code> if the end of the
	 *             file is reached.
	 * @exception IOException if an I/O error occurs.
	 */
	@Override
	public synchronized int read() throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			File.throwClosedException();
		}
		int res = FileSystem.readByteNative(fd);
		return res;
	}

	/**
	 * Reads up to <code>b.length</code> bytes of data from this input
	 * stream into an array of bytes. This method blocks until some input
	 * is available.
	 *
	 * @param b the buffer into which the data is read.
	 * @return     the total number of bytes read into the buffer, or
	 *             <code>-1</code> if there is no more data because the end of
	 *             the file has been reached.
	 * @exception IOException if an I/O error occurs.
	 */
	@Override
	public int read(byte[] b) throws IOException {
		return read(this.fd, b, 0, b.length, false);
	}

	/**
	 * Reads up to <code>len</code> bytes of data from this input stream
	 * into an array of bytes. If <code>len</code> is not zero, the method
	 * blocks until some input is available; otherwise, no
	 * bytes are read and <code>0</code> is returned.
	 *
	 * @param b   the buffer into which the data is read.
	 * @param off the start offset in the destination array <code>b</code>
	 * @param len the maximum number of bytes read.
	 * @return     the total number of bytes read into the buffer, or
	 *             <code>-1</code> if there is no more data because the end of
	 *             the file has been reached.
	 * @exception NullPointerException      If <code>b</code> is <code>null</code>.
	 * @exception IndexOutOfBoundsException If <code>off</code> is negative,
	 * <code>len</code> is negative, or <code>len</code> is greater than
	 *                                      <code>b.length - off</code>
	 * @exception IOException               if an I/O error occurs.
	 */
	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		return read(this.fd, b, off, len, false);
	}

	/**
	 * Reads as much as possible number of requested bytes.
	 *
	 * @param fd    the file descriptor.
	 * @param b     the buffer into which the data is read.
	 * @param off   the start offset in the destination array <code>b</code>
	 * @param len   the maximum number of bytes read.
	 * @param fully <code>true</code> if read must block until <code>len</code>
	 *              bytes are read, <code>false</code> otherwise.
	 * @return the total number of bytes read into the buffer, or <code>-1</code> if
	 *         there is no more data because the end of the file has been reached
	 *         and <code>fully</code> is <code>false</code>.
	 * @throws IOException  if an underlying IO error occurs while reading or if the
	 *                      stream is closed
	 * @throws EOFException if the end of stream is reached and <code>fully</code>
	 *                      is <code>true</code>.
	 */
	protected synchronized static int read(int fd, byte[] b, int off, int len, boolean fully) throws IOException {
		if (fd == -1) {
			File.throwClosedException();
		}

		ArrayTools.checkBounds(b, off, len);

		if (len == 0) {
			return 0;
		}

		int totalBytesRead = 0;

		// read blocks until at least one byte is available
		do {
			int currentReadSize = FileSystem.readNative(fd, b, off, len);
			if (currentReadSize == FileSystem.LLFS_EOF) {
				// end of file
				if (fully) {
					throw new EOFException();
				} else {
					break;
				}
			}
			totalBytesRead += currentReadSize;
			off += currentReadSize;
			len -= currentReadSize;

		} while (len > 0 && (FileSystem.availableNative(fd) > 0 || fully));

		if (totalBytesRead == 0) {
			// end of file reach
			totalBytesRead = -1;
		}

		return totalBytesRead;
	}

	/**
	 * Skips over and discards <code>n</code> bytes of data from the
	 * input stream.
	 *
	 * <p>The <code>skip</code> method may, for a variety of
	 * reasons, end up skipping over some smaller number of bytes,
	 * possibly <code>0</code>. If <code>n</code> is negative, the method
	 * will try to skip backwards. In case the backing file does not support
	 * backward skip at its current position, an <code>IOException</code> is
	 * thrown. The actual number of bytes skipped is returned. If it skips
	 * forwards, it returns a positive value. If it skips backwards, it
	 * returns a negative value.
	 *
	 * <p>This method may skip more bytes than what are remaining in the
	 * backing file. This produces no exception and the number of bytes skipped
	 * may include some number of bytes that were beyond the EOF of the
	 * backing file. Attempting to read from the stream after skipping past
	 * the end will result in -1 indicating the end of the file.
	 *
	 * @param n the number of bytes to be skipped.
	 * @return the actual number of bytes skipped.
	 * @exception  IOException  if n is negative, if the stream does not
	 *             support seek, or if an I/O error occurs.
	 */
	@Override
	public synchronized long skip(long n) throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			File.throwClosedException();
		}

		long position = FileSystem.getFilePointerNative(fd);
		long newPosition = saturatingAdd(position, n);
		FileSystem.seekNative(fd, newPosition);
		return n;
	}

	/*
	 * Add two long values without overflowing or underflowing
	 */
	private long saturatingAdd(long x, long y) {
		if (x == 0 || y == 0 || (x > 0 ^ y > 0)) {
			// zero+N or one pos, another neg = no problems
			return x + y;
		} else if (x > 0) {
			// both pos, can only overflow
			return Long.MAX_VALUE - x < y ? Long.MAX_VALUE : x + y;
		} else {
			// both neg, can only underflow
			return Long.MIN_VALUE - x > y ? Long.MIN_VALUE : x + y;
		}
	}

	/**
	 * Returns an estimate of the number of remaining bytes that can be read (or
	 * skipped over) from this input stream without blocking by the next
	 * invocation of a method for this input stream. Returns 0 when the file
	 * position is beyond EOF. The next invocation might be the same thread
	 * or another thread. A single read or skip of this many bytes will not
	 * block, but may read or skip fewer bytes.
	 *
	 * <p> In some cases, a non-blocking read (or skip) may appear to be
	 * blocked when it is merely slow, for example when reading large
	 * files over slow networks.
	 *
	 * @return     an estimate of the number of remaining bytes that can be read
	 *             (or skipped over) from this input stream without blocking.
	 * @exception IOException if this file input stream has been closed by calling
	 *                        {@code close} or an I/O error occurs.
	 */
	@Override
	public synchronized int available() throws IOException {
		int fd = this.fd;
		if (fd == -1) {
			File.throwClosedException();
		}
		int res = FileSystem.availableNative(fd);
		return res;
	}

	/**
	 * Closes this file input stream and releases any system resources
	 * associated with the stream.
	 *
	 * <p> If this stream has an associated channel then the channel is closed
	 * as well.
	 *
	 * @exception IOException if an I/O error occurs.
	 *
	 * @revised 1.4
	 * @spec JSR-51
	 */
	@Override
	public void close() throws IOException {
		synchronized (this) {
			reclaim();
		}
	}

	@Override
	public Object getSource() {
		return this;
	}

	@Override
	public void reclaim() {
		// synchronize prevent close and reclaim from being scheduled at the same time causing a double closeNative()
		synchronized (this.closesync) {
			int fd = this.fd;
			if (fd == -1) {
				return; // Already closed
			}
			this.fd = -1;

			try {
				FileSystem.closeNative(fd);
			} catch (IOException e) {
				// no-op close quitly
			}
			// file resource closed, remove it from the resource manager
			ResourceManager rm = ResourceManager.getResourceManager();
			if (rm != null) {
				rm.resourceReclaimed(this);
			}
		}
	}
}
