/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2019, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.is2t.java.io;

import java.io.File;
import java.io.FilePermission;
import java.io.IOException;

public class UnixLikeFileSystem extends FileSystem {
	private static final String rootDir = System.getProperty("root.dir", "/");
	private static final boolean caseSensitive = System.getProperty("case.sensitivity")
			.equalsIgnoreCase("caseInsensitive") ? false : true;

	/*
	 * A normal Unix pathname contains no duplicate separators and does not end with
	 * a separator. It may be the empty string.
	 */

	/* Normalize the given pathname, whose length is len, starting at the given
    offset; everything before this offset is already normal. */
	private String normalize(String pathname, int len, int off) {
		if (len == 0) {
			return pathname;
		}
		int n = len;
		while ((n > 0) && (pathname.charAt(n - 1) == File.separatorChar)) {
			n--;
		}
		if (n == 0) {
			return rootDir;
		}
		StringBuilder sb = new StringBuilder(pathname.length());
		if (off > 0) {
			sb.append(pathname.substring(0, off));
		}
		char prevChar = 0;
		for (int i = off; i < n; i++) {
			char c = pathname.charAt(i);
			if ((prevChar == File.separatorChar) && (c == File.separatorChar)) {
				continue;
			}
			sb.append(c);
			prevChar = c;
		}
		return sb.toString();
	}
	/* Check that the given pathname is normal.  If not, invoke the real
    normalizer on the part of the pathname that requires normalization.
    This way we iterate through the whole pathname string only once. */
	@Override
	public String normalize(String pathname) {
		int n = pathname.length();
		char prevChar = 0;
		for (int i = 0; i < n; i++) {
			char c = pathname.charAt(i);
			if ((prevChar == File.separatorChar) && (c == File.separatorChar)) {
				return normalize(pathname, n, i - 1);
			}
			prevChar = c;
		}
		if (prevChar == File.separatorChar) {
			return normalize(pathname, n, n - 1);
		}
		return pathname;
	}

	@Override
	public int prefixLength(String pathname) {
		if (pathname.length() == 0) {
			return 0;
		}
		return (pathname.charAt(0) == File.separatorChar) ? 1 : 0;
	}

	@Override
	public String resolve(String parent, String child) {
		if (child.equals("")) {
			return parent;
		}
		if (child.charAt(0) == File.separatorChar) {
			if (parent.equals(rootDir)) {
				return child;
			}
			return parent + child;
		}
		if (parent.equals(rootDir)) {
			return parent + child;
		}
		return parent + File.separatorChar + child;
	}

	@Override
	public String resolve(File f) {
		if (isAbsolute(f)) {
			return f.getPath();
		}
		return resolve(System.getProperty("user.dir"), f.getPath());
	}

	@Override
	public boolean isAbsolute(File f) {
		return (f.prefixLength != 0);
	}

	@Override
	public int compare(String path1, String path2) {
		if (caseSensitive) {
			return path1.compareTo(path2);
		} else {
			return path1.compareToIgnoreCase(path2);
		}
	}

	@Override
	public int hashCode(File f) {
		if (caseSensitive) {
			return f.getPath().hashCode() ^ 1234321;
		} else {
			/* Could make this more efficient: String.hashCodeIgnoreCase */
			return f.getPath().toLowerCase().hashCode() ^ 1234321;
		}
	}

	@Override
	public File[] listRoots() {
		try {
			FilePermission.checkRead(rootDir);
			return new File[] { new File(rootDir) };
		} catch (SecurityException x) {
			return new File[0];
		}
	}

	@Override
	public boolean isRoot(String absolutePath) {
		return rootDir.equals(absolutePath);
	}

	/**
	 * A simple implementation of the canonicalize() method that is implemented in Java.
	 */
	@Override
	public String canonicalize(String absolutePath) throws IOException {
		char separatorChar = File.separatorChar;
		char[] path = absolutePath.toCharArray();
		int pathLength = path.length;

		// Remove all the '/./'
		for (int i = 0; i<pathLength-1; i++) {
			// Search for "//" or "/./" or "/../"
			if(path[i]==separatorChar)
			{
				// Search for "//"
				if(path[i+1]==separatorChar){
					//remove the duplicated separator "/a//b" -> "/a/b"
					pathLength = deleteChars(path, pathLength, i, i+1);
					--i;
				}
				// Search for "/./" or "/../"
				else if(path[i+1]=='.'){
					//Test if path ends with "/." or if path contains "/./"
					if(i==pathLength-2 || path[i+2]==separatorChar){
						pathLength = deleteChars(path, pathLength, i, i+2);
						--i;
					}
					//Search for "/../"
					else if(path[i+2]=='.'){
						//Test if path ends with "/.." or if path contains "/../"
						if(i==pathLength-3 || path[i+3]==separatorChar){
							int parentDirIndex = parentDirIndex(path, i);
							pathLength = deleteChars(path, pathLength, parentDirIndex, i+3);
							i=parentDirIndex-1;
						}
					}
				}
			}
		}

		if(pathLength == 0){
			// Keep at least the '/'
			return File.separator;
		}

		if(absolutePath.length() != pathLength){
			return new String(path, 0, pathLength);
		}
		else {
			// No changes done on the string
			return absolutePath;
		}
	}

	/**
	 * Retrieve the index of the directory defined before the given index.<p>
	 * For example, for the path "/a/b/c", if you want to get the start
	 * index of the parent of "c", just do <code>parentDirIndex("/a/b/c", 4)</code>,
	 * it will return 2.<p>
	 * This method assumes that:
	 * <ul>
	 *  <li>the path starts with a path separator,</li>
	 *  <li>the path does not contains double separator ("//") from 0 to dirIndex</li>
	 * </ul>
	 *
	 * @param path char array
	 * @param dirIndex index of the directory in the path
	 * @return index of the parent directory of the directory pointed by dirIndex
	 */
	private int parentDirIndex(char[] path, int dirIndex) {
		char separator = File.separatorChar;
		while(--dirIndex >= 0){
			if(path[dirIndex] == separator){
				return dirIndex;
			}
		}
		return 0;
	}

	/**
	 * In the given char array, deletes the chars from startIndex to endIndex excluded and compacts the trailing chars.<p>
	 * @param chars the array in which the chars must be deleted
	 * @param length length of the array (may be lower than chars.length)
	 * @param startIndex index of the first char to remove
	 * @param endIndex end index exclusive
	 * @return the new length of the array
	 */
	private int deleteChars(char[] chars, int length, int startIndex, int endIndex) {
		System.arraycopy(chars, endIndex, chars, startIndex, length-endIndex);
		return length - (endIndex-startIndex);
	}

}
