/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2019, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.is2t.java.io;

import java.io.File;
import java.io.IOException;

import ej.sni.SNI;

public class UnixFileSystem extends UnixLikeFileSystem {

	@Override
	public String canonicalize(String absolutePath) throws IOException {
		byte[] pathByte = toCString(absolutePath);
		final int maxFilePathLength = FileSystem.getMaxPathLength();
		byte[] canonicalizePathByte = new byte[Math.max(maxFilePathLength, pathByte.length)];

		try {
			canonicalizeNative(pathByte, canonicalizePathByte, canonicalizePathByte.length);
			return SNI.toJavaString(canonicalizePathByte);
		}
		catch(IOException e){
			// Native implementation of canonicalize fail. This may occurs if the file does not exists.
			if(!File.exists(absolutePath)){
				// try a java implementation when the file does not exist.
				return super.canonicalize(absolutePath);
			}
			else {
				throw e;
			}
		}
	}

	/**
	 * Convert a pathname to canonical form.  The input path (path) is assumed to contain
	 * no duplicate slashes. The result is stored in canonicalizePath.
	 * On POSIX we can use realpath() to do this work.<p>
	 * This method may not throw an exception if the file referenced by the given path
	 * does not exist.
	 *
	 * @param path
	 * 			path to canonicalize
	 *
	 * @param canonicalizePath
	 * 			buffer to fill with the canonicalized path
	 *
	 * @param canonicalizePathLength
	 * 			length of canonicalizePath
	 *
	 * @throws IOException on error.
	 */
	private static native void canonicalizeNative(byte[] path, byte[] canonicalizePath, int canonicalizePathLength) throws IOException;


}
