/*
 * Copyright (c) 1994, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2021, MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.is2t.java.io;

import java.io.File;
import java.io.IOException;

import ej.bon.ByteArray;
import ej.sni.NativeException;
import ej.sni.SNI;

/**
 * FileSystem interface for fs-embedded-API Use to support new file system.
 */
abstract public class FileSystem {
	//WARNING: classname known by the launch scripts

	/** Opening mode Read for the channel. */
	public static final char FILE_MODE_READ = 'R';
	/** Opening mode Write for the channel. */
	public static final char FILE_MODE_WRITE = 'W';
	/** Opening mode Read Write for the channel. */
	public static final char FILE_MODE_READ_WRITE = 'B';
	/** Opening mode Read Write for the channel for 'rwd' sync */
	public static final char FILE_MODE_READ_WRITE_DSYNC = 'D';
	/** Opening mode Read Write for the channel. for 'rws' sync */
	public static final char FILE_MODE_READ_WRITE_SYNC = 'S';
	/** Opening mode Append for the channel. */
	public static final char FILE_MODE_APPEND = 'A';


	public static final int ACCESS_READ    = 0x04;//Native code known
	public static final int ACCESS_WRITE   = 0x02;//Native code known
	public static final int ACCESS_EXECUTE = 0x01;//Native code known

	// Possible values given to getSpaceNative()
	public static final int LLFS_FREE_SPACE = 0;
	public static final int LLFS_TOTAL_SPACE = 1;
	public static final int LLFS_USABLE_SPACE = 2;

	/**
	 * Returned native values
	 */
	public static final int LLFS_OK = 0;
	public static final int LLFS_EOF = -1;
	public static final int LLFS_NOK = -2;
	public static final int LLFS_NOT_CREATED = -3;

	/**
	 * Permission checks
	 */
	public static final int LLFS_PERMISSION_OWNER_ONLY = 0;
	public static final int LLFS_PERMISSION_ALL = 1;
	public static final int LLFS_PERMISSION_ENABLE = 0;
	public static final int LLFS_PERMISSION_DISABLE = 1;

	/** sizeof(LLFS_date_t). See LLAPI */
	public static final int LLFS_date_t_sizeof = 7 * ByteArray.INT_SIZE;
	// Offset of the <code>LLFS_date_t</code> structure fields
	public static final int LLFS_date_t_offsetof_YEAR = 0;
	public static final int LLFS_date_t_offsetof_MONTH = LLFS_date_t_offsetof_YEAR + ByteArray.INT_SIZE;
	public static final int LLFS_date_t_offsetof_DAY = LLFS_date_t_offsetof_MONTH + ByteArray.INT_SIZE;
	public static final int LLFS_date_t_offsetof_HOUR = LLFS_date_t_offsetof_DAY + ByteArray.INT_SIZE;
	public static final int LLFS_date_t_offsetof_MINUTE = LLFS_date_t_offsetof_HOUR + ByteArray.INT_SIZE;
	public static final int LLFS_date_t_offsetof_SECOND = LLFS_date_t_offsetof_MINUTE + ByteArray.INT_SIZE;
	public static final int LLFS_date_t_offsetof_MILLISECOND = LLFS_date_t_offsetof_SECOND + ByteArray.INT_SIZE;

	/**
	 * Creates a byte array and calls {@link SNI#toCString(String, byte[])}.
	 */
	public static byte[] toCString(String str) {
		byte[] cString = new byte[str.length() + 1];
		SNI.toCString(str, cString);
		return cString;
	}

	/**
	 * Convert the given pathname string to normal form.  If the string is
	 * already in normal form then it is simply returned.
	 */
	public abstract String normalize(String path);

	/**
	 * Compute the length of this pathname string's prefix.  The pathname
	 * string must be in normal form.
	 */
	public abstract int prefixLength(String path);

	/**
	 * Resolve the child pathname string against the parent.
	 * Both strings must be in normal form, and the result
	 * will be in normal form.
	 */
	public abstract String resolve(String parent, String child);

	/**
	 * Resolve the given abstract pathname into absolute form.  Invoked by the
	 * getAbsolutePath and getCanonicalPath methods in the File class.
	 */
	public abstract String resolve(File f);

	public abstract String canonicalize(String absolutePath) throws IOException;


	/* -- Path operations -- */

	/**
	 * Tell whether or not the given abstract pathname is absolute.
	 */
	public abstract boolean isAbsolute(File f);

	/**
	 * Compare two pathnames strings lexicographically.
	 *
	 * @param path1
	 *            the reference pathname string
	 * @param path2
	 *            another pathname string to compare to the reference
	 * @return a negative number if <code>reference</code> is less than <code>another</code>, 0 if they are equal, a positive number if
	 *         <code>reference</code> is greater than <code>another</code>.
	 */
	public abstract int compare(String path1, String path2);

	/**
	 * Compute the hash code of an abstract pathname.
	 */
	public abstract int hashCode(File f);

	/**
	 * List the available filesystem roots. A particular Java platform may
	 * support zero or more hierarchically-organized file systems. Each file
	 * system has a root directory from which all other files in that file
	 * system can be reached. Windows platforms, for example, have a root
	 * directory for each active drive; UNIX platforms have a single root
	 * directory, namely "/". The set of available filesystem roots is affected
	 * by various system-level operations such as the insertion or ejection of
	 * removable media and the disconnecting or unmounting of physical or
	 * virtual disk drives. This method returns an array of File objects that
	 * denote the root directories of the available filesystem roots. It is
	 * guaranteed that the canonical pathname of any file physically present on
	 * the local machine will begin with one of the roots returned by this
	 * method. The canonical pathname of a file that resides on some other
	 * machine and is accessed via a remote-filesystem protocol such as SMB or
	 * NFS may or may not begin with one of the roots returned by this method.
	 * If the pathname of a remote file is syntactically indistinguishable from
	 * the pathname of a local file then it will begin with one of the roots
	 * returned by this method. Thus, for example, File objects denoting the
	 * root directories of the mapped network drives of a Windows platform will
	 * be returned by this method, while File objects containing UNC pathnames
	 * will not be returned by this method. Unlike most methods in this class,
	 * this method does not throw security exceptions. If a security manager
	 * exists and its SecurityManager.checkRead(java.lang.String) method denies
	 * read access to a particular root directory, then that directory will not
	 * appear in the result.
	 *
	 * @return An array of File objects denoting the available filesystem roots,
	 *         or null if the set of roots could not be determined. The array
	 *         will be empty if there are no filesystem roots.
	 */
	public abstract File[] listRoots();

	/**
	 * Returns true if the given absolute path is a root, false otherwise.
	 *
	 * @param absolutePath
	 *            the path to test
	 */
	public abstract boolean isRoot(String absolutePath);


	/*
	 * Native methods.
	 */

	/**
	 * Initializes the C part of the FS library such as the file system
	 * middleware, the management tasks and the drivers.
	 *
	 * @throws NativeException on error.
	 */
	public static native void initializeNative() throws NativeException;

	/**
	 * Get last modified date of the file. The date must be stored in given
	 * array (see LLFS_date_t in LLAPI).
	 *
	 * @param path
	 *            absolute path of file
	 * @param date
	 *            the array where store the date (sse LLFS_date_t in LLAPI)
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int lastModifiedNative(byte[] path, byte[] date);

	/**
	 * Update file as read only.
	 *
	 * @param path
	 *            absolute path of file
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int setReadOnlyNative(byte[] path);

	/**
	 * Create a new empty file with the given pathname.
	 * <p>
	 * When action is {@link #LLFS_EVENT_RETRIEVE}, return {@link #LLFS_OK} if
	 * the file was created, {@link #LLFS_NOT_CREATED} if a file or directory
	 * with the given pathname already exists, throw an {@link IOException} if an
	 * I/O error occurs.
	 *
	 * @param path
	 *            absolute path of file
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOT_CREATED}
	 *
	 * @throws IOException on error.
	 */
	public static native int createNative(byte[] path) throws IOException;

	/**
	 * Opens a directory in order to read from it later. Framework ensures only one
	 * directory is opened at the same time closing current directory without
	 * opening a new one. The returned value is an positive integer used as ID to
	 * identify the opened directory. This ID will be used later by read and close
	 * methods.
	 *
	 * @param path
	 *            absolute path of directory
	 * @return an integer out of the range
	 *         <code>[{@link #LLFS_NOT_CREATED}-{@link #LLFS_OK}]</code> or
	 *         {@link #LLFS_NOK}
	 */
	public static native int opendirNative(byte[] path);

	/**
	 * Specify the next available file in directory. Returns {@link #LLFS_NOK}
	 * when no more file is available.
	 *
	 * @param dirID
	 *            the directory ID
	 * @param path
	 *            path where store the next available file
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int readdirNative(int dirID, byte[] path);

	/**
	 * Close the directory.
	 *
	 * @param dirID
	 *            the directory ID
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int closedirNative(int dirID);

	/**
	 * Rename a file.
	 *
	 * @param path
	 *            absolute path of file to modify
	 * @param newPath
	 *            new absolute path of file
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int renameToNative(byte[] path, byte[] newPath);

	/**
	 * Get the length, in bytes, of the file denoted by path, or 0L if the file
	 * does not exist. Some operating systems may return 0L for pathnames
	 * denoting system-dependent entities such as devices or pipes
	 *
	 * @param path
	 *            absolute path of file to test
	 * @return The length of the file or {@link #LLFS_NOK} on error.
	 */
	public static native long lengthNative(byte[] path);

	/**
	 * Returns {@link #LLFS_OK} if and only if the file or directory denoted by
	 * this abstract pathname exists, {@link #LLFS_NOK} otherwise.
	 *
	 * @param path
	 *            absolute path of file to test
	 * @return {@link #LLFS_OK} when file exists, {@link #LLFS_NOK} otherwise.
	 */
	public static native int existNative(byte[] path);

	/**
	 * Create the directory.
	 *
	 * @param path
	 *            absolute path of directory with the last entity to create
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int mkdirNative(byte[] path);

	/**
	 * Tests whether the file named by this abstract pathname is a hidden file.
	 * The exact definition of hidden is system-dependent. On UNIX systems, a
	 * file is considered to be hidden if its name begins with a period
	 * character ('.'). On Microsoft Windows systems, a file is considered to be
	 * hidden if it has been marked as such in the filesystem.
	 *
	 * @param path
	 *            absolute path of file to test.
	 * @return {@link #LLFS_OK} if and only if the file denoted by this path is
	 *         hidden according to the conventions of the underlying platform,
	 *         {@link #LLFS_NOK} otherwise
	 */
	public static native int isHiddenNative(byte[] path);

	/**
	 * Check if the file denoted by this abstract pathname exists and is a
	 * directory
	 *
	 * @param path
	 *            absolute path of file to test
	 * @return {@link #LLFS_OK} (file is a directory) or {@link #LLFS_NOK}
	 */
	public static native int isDirectoryNative(byte[] path);

	/**
	 * Check if the file denoted by this abstract pathname exists and is a file
	 *
	 * @param path
	 *            absolute path of file to test
	 * @return {@link #LLFS_OK} (file is not a directory) or {@link #LLFS_NOK}
	 */
	public static native int isFileNative(byte[] path);

	/**
	 * Depending on <code>spaceType</code> returns the following values:
	 * <ul>
	 *
	 * 	<li>{@link #LLFS_FREE_SPACE}: number of unallocated bytes on the partition or
	 * 0L if the path does not name a partition</li>
	 *
	 * 	<li>{@link #LLFS_TOTAL_SPACE}: size, in bytes, of the partition or 0L if this
	 * path does not name a partition</li>
	 *
	 * 	<li>{@link #LLFS_USABLE_SPACE}: number of available bytes on the partition or
	 * 0L if the path does not name a partition. On systems where this information is
	 * not available, this method will be equivalent to a call to
	 * <code>getSpaceNative(LLFS_FREE_SPACE)</code>.</li>
	 *
	 * </ul>
	 *
	 * @param path
	 *            absolute path of file in the require partition
	 * @param spaceType
	 * 			  {@link #LLFS_FREE_SPACE}, {@link #LLFS_TOTAL_SPACE} or {@link #LLFS_USABLE_SPACE}.
	 *
	 * @return The number of bytes or {@link #LLFS_NOK} on error.
	 */
	public static native long getSpaceSizeNative(byte[] path, int spaceType);

	/**
	 * Change the last modified date of file.
	 *
	 * @param path
	 *            absolute path of file to modified
	 * @param date
	 * 		the new date (see LLFS_date_t in LLAPI)
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int setLastModifiedNative(byte[] path, byte[] date);

	/**
	 * Delete the file or directory denoted by the given absolute pathname.
	 *
	 * @param path
	 *            absolute path of file to modified
	 *
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int deleteNative(byte[] path);

	/**
	 * Check whether the file or directory denoted by the given abstract
	 * pathname may be accessed by this process. The second argument specifies
	 * which access, {@link #ACCESS_READ}, {@link #ACCESS_WRITE} or
	 * {@link #ACCESS_EXECUTE}, to check.
	 *
	 * @param path
	 *            absolute path of file to modified
	 * @param access
	 *            {@link #ACCESS_READ}, {@link #ACCESS_WRITE} or
	 *            {@link #ACCESS_EXECUTE}
	 *
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int checkAccessNative(byte[] path, int access);

	/**
	 * Set on or off the access permission (to owner only or to all) to the file
	 * or directory denoted by the given abstract pathname.
	 *
	 * @param path
	 *            absolute path of file to modified
	 * @param access
	 *            {@link #ACCESS_READ}, {@link #ACCESS_WRITE} or
	 *            {@link #ACCESS_EXECUTE}
	 * @param enable
	 *            {@link #LLFS_PERMISSION_ENABLE} or
	 *            {@link #LLFS_PERMISSION_DISABLE}
	 * @param owner
	 *            {@link #LLFS_PERMISSION_OWNER_ONLY} or
	 *            {@link #LLFS_PERMISSION_ALL}
	 *
	 * @return {@link #LLFS_OK} (no error) or {@link #LLFS_NOK}
	 */
	public static native int setPermissionNative(byte[] path, int access, int enable, int owner);


	/**
	 * Open a file according given mode. The returned value is a positive
	 * integer used to identify the file later.
	 *
	 * @param path
	 *            absolute path of file to open
	 * @param mode
	 *            opening mode : {@link #LLFS_FILE_MODE_READ},
	 *            {@link #LLFS_FILE_MODE_WRITE} or
	 *            {@link #LLFS_FILE_MODE_APPEND}
	 *
	 * @return a unique ID
	 *
	 * @throws IOException on error.
	 */
	public static native int openNative(byte[] path, char mode) throws IOException;

	/**
	 * Read one byte from the file.
	 *
	 * @param fileID
	 *            file identifier
	 * @return the read byte as an unsigned integer or {@link #LLFS_EOF} on EOF.
	 *
	 * @throws IOException on error.
	 */
	public static native int readByteNative(int fd) throws IOException;

	/**
	 * Reads data from the channel.
	 *
	 * @param fileID
	 *            file identifier
	 * @param data
	 *            pointer to byte array
	 * @param offset
	 * 			  the start offset in the destination array <code>data</code>
	 * @param length
	 *            number of data to read
	 * @return number of bytes read or {@link #LLFS_EOF} on EOF.
	 *
	 * @throws IOException on error.
	 */
	public static native int readNative(int fd, byte[] data, int offset, int length) throws IOException;

	/**
	 * Sets the file-pointer offset, measured from the beginning of this file, at which the next read or write occurs.
	 * The offset may be set beyond the end of the file. Setting the offset beyond the end of the file does not change
	 * the file length. The file length will change only by writing after the offset has been set beyond the end of the
	 * file.
	 *
	 * @param pos
	 *            the offset position, measured in bytes from the beginning of the file, at which to set the file
	 *            pointer.
	 * @exception IOException
	 *                if {@code pos} is less than {@code 0} or if an I/O error occurs.
	 */
	public static native void seekNative(int fd, long pos) throws IOException;

	/**
	 * Returns the current offset of the give file descriptor.
	 *
	 * @param fd
	 *            file identifier
	 *
	 * @return the offset from the beginning of the file, in bytes, at which the next read or write occurs.
	 * @exception IOException
	 *                if an I/O error occurs.
	 */
	public static native long getFilePointerNative(int fd) throws IOException;

	/**
	 * Returns the length of this file.
	 *
	 * @param fd file identifier
	 * @return the length of this file, measured in bytes.
	 * @exception IOException If an I/O error occurs
	 */
	public static native long lengthWithFdNative(int fd) throws IOException;

	/**
	 * Sets the length of this file.
	 * <p>
	 * If the present length of the file as returned by the {@code length} method is
	 * greater than the {@code newLength} argument then the file will be truncated.
	 * In this case, if the file offset as returned by the {@code getFilePointer}
	 * method is greater than {@code newLength} then after this method returns the
	 * offset will be equal to {@code newLength}.
	 * <p>
	 * If the present length of the file as returned by the {@code length} method is
	 * smaller than the {@code newLength} argument then the file will be extended.
	 * In this case, the contents of the extended portion of the file are not
	 * defined.
	 *
	 * @param fd        file identifier
	 * @param newLength The desired length of the file
	 * @exception IOException If an I/O error occurs
	 */
	public static native void setLengthNative(int fd, long newLength) throws IOException;

	/**
	 * Returns an estimate of the number of remaining bytes that can be read (or
	 * skipped over) from this channel without blocking by the next invocation
	 * of a method for this channel. Returns 0 when the file position is beyond
	 * EOF.
	 *
	 * @param fileID
	 *            file identifier
	 * @return an estimate of the number of remaining bytes that can be read (or
	 *         skipped over) from this channel without blocking.
	 *
	 * @throws IOException on error.
	 */
	public static native int availableNative(int fd) throws IOException;

	/**
	 * Close the file.
	 *
	 * @param fileID
	 *            file identifier
	 *
	 * @throws IOException on error.
	 */
	public static native void closeNative(int fd) throws IOException;

	/**
	 * Writes data to the channel.
	 *
	 * @param fileID
	 *            file identifier
	 * @param data
	 *            pointer to byte array
	 * @param offset
	 *            the start offset in the data
	 * @param length
	 *            number of data to write
	 * @return The number of bytes written. Possibly zero if there was nothing to
	 *         write, for instance.
	 *
	 * @throws IOException
	 *             on error.
	 */
	public static native int writeNative(int fileID, byte[] data, int offset, int length) throws IOException;

	/**
	 * Writes the given byte in file.
	 *
	 * @param fd
	 *            file identifier
	 * @param data
	 *            the byte to write
	 *
	 * @throws IOException on error.
	 */
	public static native void writeByteNative(int fd, int data) throws IOException;

	/**
	 * Returns the maximum path length managed in native.
	 */
	public static native int getMaxPathLength();

	/**
	 * Flushes the output stream identified by the given file descriptor and forces
	 * any buffered data to be written out.
	 *
	 * @param fd
	 *            the file identifier
	 */
	public static native void flushNative(int fd);

}
